<?php
if (!defined('ABSPATH')) exit;

/**
 * Helpers comunes (permiso + empresas + sesión)
 */
function _gl_get_context_empresas() {
    global $wpdb;

    $tabla_users    = $wpdb->prefix . 'users';
    $tabla_empresas = $wpdb->prefix . 'empresas';

    $wp_user_id = get_current_user_id();
    if (!$wp_user_id) return ['error' => 'Debes iniciar sesión.'];

    $gestor = $wpdb->get_row($wpdb->prepare(
        "SELECT ID, user_email, user_rol, user_estado FROM $tabla_users WHERE ID = %d",
        $wp_user_id
    ));
    if (!$gestor || $gestor->user_rol !== 'Gestor' || $gestor->user_estado !== 'Activado') {
        return ['error' => 'No tienes permisos para gestionar lugares.'];
    }

    // Todos los CIFs del gestor
    $cifs = $wpdb->get_col($wpdb->prepare(
        "SELECT DISTINCT user_cif FROM $tabla_users 
         WHERE user_email = %s AND user_rol = 'Gestor' AND user_estado = 'Activado' AND user_cif IS NOT NULL",
        $gestor->user_email
    ));
    if (!$cifs) return ['error' => 'No tienes empresas asociadas.'];

    // Empresas activas de esos CIF
    $in = implode(',', array_fill(0, count($cifs), '%s'));
    $empresas = $wpdb->get_results($wpdb->prepare(
        "SELECT id, user_cif, razon_social, estado_empresa, IFNULL(estado_lugares,'desactivado') AS estado_lugares
         FROM $tabla_empresas
         WHERE user_cif IN ($in) AND estado_empresa = 'activado'
         ORDER BY razon_social ASC, user_cif ASC",
        ...$cifs
    ));
    if (!$empresas) return ['error' => 'No hay empresas activas asociadas.'];

    if (!session_id()) session_start();

    // Índices útiles
    $empresa_ids = [];
    $map_cif_to_empresa_id = [];
    foreach ($empresas as $e) {
        $eid = (int)$e->id;
        $empresa_ids[] = $eid;
        // Si hubiese varias empresas con el mismo CIF (no debería), nos quedamos con la primera
        if (!isset($map_cif_to_empresa_id[$e->user_cif])) {
            $map_cif_to_empresa_id[$e->user_cif] = $eid;
        }
    }

    // 1) Prioriza empresa_id explícito por POST/GET
    if (isset($_REQUEST['empresa_id'])) {
        $pid = (int) $_REQUEST['empresa_id'];
        if (in_array($pid, $empresa_ids, true)) {
            $_SESSION['empresa_id_activa_lug'] = $pid;
        }
    }
    // 2) Si viene CIF por POST/GET (como en Revisión), mapea a empresa_id
    elseif (!empty($_REQUEST['cif'])) {
        $cif_req = sanitize_text_field($_REQUEST['cif']);
        if (isset($map_cif_to_empresa_id[$cif_req])) {
            $_SESSION['empresa_id_activa_lug'] = $map_cif_to_empresa_id[$cif_req];
        }
    }
    // 3) Si hay cif_activo en sesión (puesto por tu selector de arriba), úsalo
    elseif (!empty($_SESSION['cif_activo']) && isset($map_cif_to_empresa_id[$_SESSION['cif_activo']])) {
        $_SESSION['empresa_id_activa_lug'] = $map_cif_to_empresa_id[$_SESSION['cif_activo']];
    }

    // 4) Fallback a la primera empresa activa
    $empresa_id = isset($_SESSION['empresa_id_activa_lug']) && in_array((int)$_SESSION['empresa_id_activa_lug'], $empresa_ids, true)
        ? (int)$_SESSION['empresa_id_activa_lug']
        : (int)$empresas[0]->id;

    // Empresa seleccionada
    $empresa_sel = null;
    foreach ($empresas as $e) {
        if ((int)$e->id === $empresa_id) { $empresa_sel = $e; break; }
    }
    if (!$empresa_sel) return ['error' => 'No se pudo determinar la empresa seleccionada.'];

    return [
        'gestor'      => $gestor,
        'empresas'    => $empresas,
        'empresa_sel' => $empresa_sel,
        'empresa_id'  => $empresa_id,
        'empresa_ids' => $empresa_ids,
    ];
}



/**
 * SHORTCODE 1: Formulario para AÑADIR lugares
 * [gestion_lugares_add]
 */
function shortcode_gestion_lugares_add() {
    global $wpdb;
    $ctx = _gl_get_context_empresas();
    if (isset($ctx['error'])) return '<p class="pure-alert" style="text-align:center;">'.$ctx['error'].'</p>';

    $empresa_sel = $ctx['empresa_sel'];
    $empresa_id  = $ctx['empresa_id'];

// Si la empresa no tiene lugares activados, no mostramos nada
if (empty($empresa_sel->estado_lugares) || $empresa_sel->estado_lugares !== 'activado') {
    return '';
}


    $tabla_lugares = $wpdb->prefix . 'lugares';
    $mensaje = '';

    // Procesar alta
    if (!empty($_POST['accion_add_lugar'])) {
        if (!isset($_POST['lugares_add_nonce_field']) ||
            !wp_verify_nonce($_POST['lugares_add_nonce_field'], 'lugares_add_nonce')) {
            return '<p style="color:red;">❌ Seguridad fallida.</p>';
        }

        $empresa_id_post = (int)($_POST['empresa_id'] ?? 0);
        if ($empresa_id_post !== $empresa_id) {
            return '<p style="color:red;">❌ Empresa no válida.</p>';
        }

        $nombre_lugar = trim(sanitize_text_field($_POST['nombre_lugar'] ?? ''));
        $direccion    = isset($_POST['direccion']) ? wp_kses_post($_POST['direccion']) : null;

        if ($nombre_lugar === '') {
            $mensaje = '<div style="color:#c00;">Debes indicar un nombre para el lugar.</div>';
        } else {
            $ok = $wpdb->insert($tabla_lugares, [
                'empresa_id'    => $empresa_id,
                'nombre_lugar'  => $nombre_lugar,
                'direccion'     => $direccion,
                'qr_code'       => null,
                'estado'        => 'activado',
                'created_at'    => current_time('mysql'),
                'updated_at'    => current_time('mysql'),
            ]);
            $mensaje = $ok ? '<div style="color:green;">✅ Lugar añadido.</div>'
                           : '<div style="color:#c00;">❌ Error al añadir lugar: '.esc_html($wpdb->last_error).'</div>';
        }
    }

    ob_start();
    ?>
    <div class="gestion-lugares-add" style="max-width: 800px; margin: 0 auto;">
        <h3>Añadir lugar</h3>


        <?php if ($mensaje) echo $mensaje; ?>

        <div style="background:#f9f9f9;padding:12px;border-radius:6px;margin:12px 0;">
            <form method="post" class="pure-form pure-form-stacked">
                <?php wp_nonce_field('lugares_add_nonce', 'lugares_add_nonce_field'); ?>
                <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id); ?>">
                <input type="hidden" name="accion_add_lugar" value="1">

                <label>Nombre del lugar *</label>
                <input type="text" name="nombre_lugar" required style="width:100%;">

                <label style="margin-top:8px;">Dirección (opcional)</label>
                <textarea name="direccion" rows="2" style="width:100%;"></textarea>

                <button type="submit" class="pure-button pure-button-primary" style="margin-top:10px;">➕ Añadir lugar</button>
            </form>
        </div>
    </div>
    <?php
    return ob_get_clean();
}

add_shortcode('gestion_lugares_add', 'shortcode_gestion_lugares_add');

/**
 * SHORTCODE 2: Listado + filtro por estado + activar/desactivar
 * [gestion_lugares_lista]
 */
function shortcode_gestion_lugares_lista() {
    global $wpdb;
    $ctx = _gl_get_context_empresas();
    if (isset($ctx['error'])) return '<p class="pure-alert" style="text-align:center;">'.$ctx['error'].'</p>';

    $empresa_sel = $ctx['empresa_sel'];
    $empresa_id  = $ctx['empresa_id'];
    // $empresas NO se usa ya para pintar select

// Si la empresa no tiene lugares activados, no mostramos nada
if (empty($empresa_sel->estado_lugares) || $empresa_sel->estado_lugares !== 'activado') {
    return '';
}




    $tabla_lugares = $wpdb->prefix . 'lugares';
    $mensaje = '';

    // Filtro por estado (activado por defecto)
    $valid_estados = ['activado','desactivado','todos'];
    if (!session_id()) session_start();
    if (isset($_POST['estado_filtro']) && in_array($_POST['estado_filtro'], $valid_estados, true)) {
        $_SESSION['estado_filtro_lugares'] = $_POST['estado_filtro'];
    }
    $estado_filtro = $_SESSION['estado_filtro_lugares'] ?? 'activado';

    // Toggle estado
    if (!empty($_POST['accion_toggle_lugar'])) {
        if (!isset($_POST['lugares_toggle_nonce_field']) ||
            !wp_verify_nonce($_POST['lugares_toggle_nonce_field'], 'lugares_toggle_nonce')) {
            return '<p style="color:red;">❌ Seguridad fallida.</p>';
        }

        $empresa_id_post = (int)($_POST['empresa_id'] ?? 0);
        $lugar_id        = (int)($_POST['lugar_id'] ?? 0);
        if ($empresa_id_post !== $empresa_id || !$lugar_id) {
            return '<p style="color:red;">❌ Datos no válidos.</p>';
        }

        $lugar = $wpdb->get_row($wpdb->prepare(
            "SELECT id, estado FROM $tabla_lugares WHERE id = %d AND empresa_id = %d",
            $lugar_id, $empresa_id
        ));
        if ($lugar) {
            $nuevo_estado = ($lugar->estado === 'activado') ? 'desactivado' : 'activado';
            $ok = $wpdb->update($tabla_lugares, [
                'estado'     => $nuevo_estado,
                'updated_at' => current_time('mysql'),
            ], ['id' => $lugar_id]);
            $mensaje = $ok !== false
                ? '<div style="color:green;">✅ Estado actualizado.</div>'
                : '<div style="color:#c00;">❌ No se pudo actualizar el estado.</div>';
        } else {
            $mensaje = '<div style="color:#c00;">❌ Lugar no encontrado o no pertenece a la empresa.</div>';
        }
    }

    // Query según filtro
    if ($estado_filtro === 'todos') {
        $lugares = $wpdb->get_results($wpdb->prepare(
            "SELECT id, nombre_lugar, direccion, estado
             FROM $tabla_lugares
             WHERE empresa_id = %d
             ORDER BY nombre_lugar ASC",
            $empresa_id
        ));
    } else {
        $lugares = $wpdb->get_results($wpdb->prepare(
            "SELECT id, nombre_lugar, direccion, estado
             FROM $tabla_lugares
             WHERE empresa_id = %d AND estado = %s
             ORDER BY nombre_lugar ASC",
            $empresa_id, $estado_filtro
        ));
    }

    ob_start();
    ?>
    <div class="gestion-lugares-lista" style="max-width: 900px; margin: 0 auto;">
        <h3>Lugares</h3>

    

        <!-- Filtro estado -->
        <form method="post" class="pure-form" style="margin:8px 0;">
            <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id); ?>">
            <label for="estado_filtro">Estado:</label>
            <select name="estado_filtro" id="estado_filtro" onchange="this.form.submit()">
                <option value="activado"   <?php selected($estado_filtro, 'activado'); ?>>Activados</option>
                <option value="desactivado"<?php selected($estado_filtro, 'desactivado'); ?>>Desactivados</option>
                <option value="todos"      <?php selected($estado_filtro, 'todos'); ?>>Todos</option>
            </select>
        </form>

        <?php if ($mensaje) echo $mensaje; ?>

        <?php if ($lugares): ?>
            <table class="pure-table pure-table-bordered" style="width:100%;">
                <thead>
                    <tr>
                        <th>Nombre</th>
                        <th>Estado</th>
                        <th>Acción</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($lugares as $l): ?>
                    <tr>
                        <td>
                            <strong><?php echo esc_html($l->nombre_lugar); ?></strong>
                            <?php if (!empty($l->direccion)): ?>
                                <div style="color:#666;font-size:12px;"><?php echo esc_html(wp_strip_all_tags($l->direccion)); ?></div>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span style="padding:2px 6px;border-radius:4px;
                                background:<?php echo $l->estado==='activado' ? '#e6ffed' : '#ffe9e9'; ?>;
                                color:<?php echo $l->estado==='activado' ? '#0a7' : '#c00'; ?>;">
                                <?php echo esc_html($l->estado); ?>
                            </span>
                        </td>
                        <td style="text-align:center;">
                            <form method="post" style="display:inline;">
                                <?php wp_nonce_field('lugares_toggle_nonce', 'lugares_toggle_nonce_field'); ?>
                                <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id); ?>">
                                <input type="hidden" name="accion_toggle_lugar" value="1">
                                <input type="hidden" name="lugar_id" value="<?php echo esc_attr($l->id); ?>">
                                <?php if ($l->estado === 'activado'): ?>
                                    <button class="pure-button" style="background:#bbb;color:#fff;"
                                        onclick="return confirm('¿Desactivar este lugar?');">Desactivar</button>
                                <?php else: ?>
                                    <button class="pure-button pure-button-secondary"
                                        onclick="return confirm('¿Activar este lugar?');">Activar</button>
                                <?php endif; ?>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>No hay lugares con el estado seleccionado.</p>
        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
}

add_shortcode('gestion_lugares_lista', 'shortcode_gestion_lugares_lista');
