<?php
// Evitar acceso directo
if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

// Solo admin (o quien tenga manage_options)
if (!current_user_can('manage_options')) {
    echo '<div class="notice notice-error"><p>❌ No tienes permisos para acceder a esta página.</p></div>';
    return;
}

$mensaje_exito = '';

// 1) EMPRESAS DISPONIBLES
$empresas = $wpdb->get_results("
    SELECT id, user_cif, razon_social, estado_lugares
    FROM {$wpdb->prefix}empresas
    ORDER BY razon_social ASC, user_cif ASC
");

if (!$empresas) {
    echo '<p class="pure-alert" style="text-align:center;">No hay empresas configuradas.</p>';
    return;
}

// Empresa seleccionada
$empresa_id_activa = isset($_POST['empresa_id'])
    ? intval($_POST['empresa_id'])
    : (isset($_GET['empresa_id']) ? intval($_GET['empresa_id']) : (int)$empresas[0]->id);

$empresa_activa = null;
foreach ($empresas as $e) {
    if ((int)$e->id === $empresa_id_activa) {
        $empresa_activa = $e;
        break;
    }
}

if (!$empresa_activa) {
    $empresa_activa = $empresas[0];
    $empresa_id_activa = (int)$empresa_activa->id;
}

$user_cif = $empresa_activa->user_cif;
$empresa_tiene_lugares = ($empresa_activa->estado_lugares === 'activado');

// 2) RANGO DE FECHAS
$hoy = current_time('Y-m-d');
$fecha_inicio = isset($_POST['fecha_inicio']) ? sanitize_text_field($_POST['fecha_inicio']) : $hoy;
$fecha_fin    = isset($_POST['fecha_fin'])    ? sanitize_text_field($_POST['fecha_fin'])    : $hoy;

// 3) TRABAJADORES DE ESA EMPRESA
$trabajadores = $wpdb->get_results($wpdb->prepare(
    "SELECT ID, display_name, user_rol, user_dni
     FROM {$wpdb->prefix}users
     WHERE user_cif = %s
       AND user_rol IN ('Trabajador', 'Gestor')
       AND user_estado = 'Activado'
     ORDER BY display_name ASC",
    $user_cif
));

$trabajador_id = isset($_POST['trabajador_id']) ? intval($_POST['trabajador_id']) : 0;

// ---------------------------------------------------
// ACCIÓN: TICAR ENTRADA / SALIDA (ADMIN)
// ---------------------------------------------------
if (!empty($_POST['accion_ticar']) && !empty($trabajador_id)) {

    // Validar nonce
    if (!isset($_POST['ticar_nonce_field']) || !wp_verify_nonce($_POST['ticar_nonce_field'], 'ticar_nonce')) {
        echo '<p style="color:red;">❌ Seguridad fallida al ticar.</p>';
        return;
    }

    // Validar trabajador pertenece a la empresa
    $trab_ok = $wpdb->get_row($wpdb->prepare(
        "SELECT ID FROM {$wpdb->prefix}users 
         WHERE ID = %d AND user_cif = %s AND user_estado = 'Activado'",
        $trabajador_id, $user_cif
    ));
    if (!$trab_ok) {
        echo '<p style="color:red;">❌ El trabajador no es válido o no pertenece a esta empresa.</p>';
        return;
    }

    // ¿Hay registro abierto?
    $registro_abierto = $wpdb->get_row($wpdb->prepare(
        "SELECT id, entradaR 
         FROM mnkt_registro_horario
         WHERE user_id = %d AND cif_usuario = %s AND salida IS NULL
         ORDER BY entrada DESC
         LIMIT 1",
        $trabajador_id, $user_cif
    ));

    // Lugar seleccionado
    $lugar_id = isset($_POST['lugar_id']) ? intval($_POST['lugar_id']) : 0;

    // Si empresa usa lugares, no hay registro abierto (entrada) y no hay lugar aún → pedir lugar
    if ($empresa_tiene_lugares && !$registro_abierto && !$lugar_id) {

        $tabla_lugares = $wpdb->prefix . 'lugares'; // asumiendo tabla mnkt_lugares
        $lugares = $wpdb->get_results($wpdb->prepare(
            "SELECT id, nombre_lugar 
             FROM $tabla_lugares
             WHERE empresa_id = %d
               AND estado = 'activado'
             ORDER BY nombre_lugar ASC",
            $empresa_id_activa
        ));

        if (empty($lugares)) {
            echo '<p class="pure-alert" style="color:#c00;">No hay lugares activados para esta empresa.</p>';
            return;
        }

        ?>
        <div class="wrap">
            <h2>Seleccionar lugar para el fichaje</h2>
            <div style="background:#fff3cd; border:1px solid #ffeeba; padding:15px; border-radius:6px; max-width:600px;">
                <p><strong>Selecciona el lugar para este fichaje.</strong></p>
                <form method="post">
                    <?php wp_nonce_field('ticar_nonce', 'ticar_nonce_field'); ?>
                    <input type="hidden" name="accion_ticar" value="1">
                    <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id_activa); ?>">
                    <input type="hidden" name="trabajador_id" value="<?php echo esc_attr($trabajador_id); ?>">
                    <input type="hidden" name="fecha_inicio" value="<?php echo esc_attr($fecha_inicio); ?>">
                    <input type="hidden" name="fecha_fin" value="<?php echo esc_attr($fecha_fin); ?>">

                    <label for="lugar_id">📍 Lugar del fichaje:</label>
                    <select name="lugar_id" id="lugar_id" required>
                        <option value="">-- Selecciona un lugar --</option>
                        <?php foreach ($lugares as $l): ?>
                            <option value="<?php echo esc_attr($l->id); ?>">
                                <?php echo esc_html($l->nombre_lugar); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>

                    <p><button type="submit" class="button button-primary">✔️ Confirmar fichaje con este lugar</button></p>
                </form>
            </div>
        </div>
        <?php
        return;
    }

    // EN ADELANTE: ya tenemos registro_abierto (SALIDA) o ENTRADA con lugar
    $ahora_mysql = current_time('mysql');

    if ($registro_abierto) {
        // SALIDA
        $inicio = strtotime($registro_abierto->entradaR ?: $ahora_mysql);
        $fin    = strtotime($ahora_mysql);
        $diff   = max(0, $fin - $inicio);

        $tiempo_total = sprintf('%02d:%02d:%02d',
            floor($diff / 3600),
            floor(($diff % 3600) / 60),
            $diff % 60
        );

        $wpdb->update(
            "mnkt_registro_horario",
            [
                'salida'        => $ahora_mysql,
                'salidaR'       => $ahora_mysql,
                'tiempo_total'  => $tiempo_total,
                'tiempo_totalR' => $tiempo_total,
                'puntos'        => -100
            ],
            ['id' => $registro_abierto->id]
        );

        // Observación
        $nombre_admin = wp_get_current_user()->display_name;
        $fecha_obs    = date('Y-m-d', strtotime($ahora_mysql));
        $ts           = date('d/m/Y H:i:s', strtotime($ahora_mysql));
        $obs_txt      = "$ts [Ticar manual ADMIN ($nombre_admin)] SALIDA";

        $wpdb->insert("{$wpdb->prefix}observaciones", [
            'user_id'          => $trabajador_id,
            'cif_usuario'      => $user_cif,
            'fecha_observacion'=> $fecha_obs,
            'observaciones'    => $obs_txt
        ]);

        $mensaje_exito = '✅ Salida ticada (admin).';

    } else {
        // ENTRADA
        $datos_insert = [
            'user_id'        => $trabajador_id,
            'cif_usuario'    => $user_cif,
            'entrada'        => $ahora_mysql,
            'entradaR'       => $ahora_mysql,
            'tiempo_total'   => '00:00:00',
            'tiempo_totalR'  => '00:00:00',
            'puntos'         => -100
        ];

        if ($empresa_tiene_lugares && $lugar_id) {
            $datos_insert['lugar_id'] = $lugar_id;
        }

        $wpdb->insert("mnkt_registro_horario", $datos_insert);
        $nuevo_id = $wpdb->insert_id;

        $nombre_admin = wp_get_current_user()->display_name;
        $fecha_obs    = date('Y-m-d', strtotime($ahora_mysql));
        $ts           = date('d/m/Y H:i:s', strtotime($ahora_mysql));
        $obs_txt      = "$ts [Ticar manual ADMIN ($nombre_admin)] ENTRADA";

        $wpdb->insert("{$wpdb->prefix}observaciones", [
            'user_id'          => $trabajador_id,
            'cif_usuario'      => $user_cif,
            'fecha_observacion'=> $fecha_obs,
            'observaciones'    => $obs_txt
        ]);

        $mensaje_exito = '✅ Entrada ticada (admin).';
    }
}

// ---------------------------------------------------
// ACCIÓN: EDITAR REGISTRO (ENTRADA/ SALIDA)
// ---------------------------------------------------
if (!empty($_POST['editar_id'])) {

    if (!isset($_POST['editar_fichaje_nonce_field']) ||
        !wp_verify_nonce($_POST['editar_fichaje_nonce_field'], 'editar_fichaje_nonce')) {
        echo '<p style="color:red;">❌ Seguridad fallida al editar registro.</p>';
        return;
    }

    $id_editar = intval($_POST['editar_id']);
    $entradaR  = sanitize_text_field($_POST['entradaR']);
    $salidaR   = sanitize_text_field($_POST['salidaR']);

    $tiempo_totalR = null;
    if ($entradaR && $salidaR) {
        $inicio = strtotime($entradaR);
        $fin    = strtotime($salidaR);
        if ($inicio < $fin) {
            $segundos = $fin - $inicio;
            $tiempo_totalR = sprintf(
                '%02d:%02d:%02d',
                floor($segundos / 3600),
                floor(($segundos % 3600) / 60),
                $segundos % 60
            );
        }
    }

    // Registro original
    $registro_original = $wpdb->get_row($wpdb->prepare(
        "SELECT entradaR, salidaR FROM mnkt_registro_horario WHERE id = %d",
        $id_editar
    ));

    $datos_update = [
        'entradaR'      => $entradaR,
        'salidaR'       => $salidaR,
        'tiempo_totalR' => $tiempo_totalR
    ];

    // Si antes no había salidaR, ajustamos también salida y puntos
    if (empty($registro_original->salidaR)) {
        $datos_update['salida']       = $entradaR;
        $datos_update['puntos']       = -60;
        $datos_update['tiempo_total'] = '00:00:00';
    }

    $wpdb->update("mnkt_registro_horario", $datos_update, ['id' => $id_editar]);

    // Observación de modificación
    $trabajador       = get_userdata($trabajador_id);
    $nombre_admin     = wp_get_current_user()->display_name;
    $cif_trabajador   = $trabajador ? $trabajador->user_cif : $user_cif;

    $fecha_actual         = current_time('Y-m-d H:i:s');
    $fecha_observacion    = date('Y-m-d', strtotime($fecha_actual));
    $timestamp_formateado = date('d/m/Y H:i:s', strtotime($fecha_actual));

    $entrada_antes = !empty($registro_original->entradaR)
        ? date('d/m/Y H:i:s', strtotime($registro_original->entradaR))
        : '—';
    $salida_antes = !empty($registro_original->salidaR)
        ? date('d/m/Y H:i:s', strtotime($registro_original->salidaR))
        : '—';
    $entrada_despues = !empty($entradaR)
        ? date('d/m/Y H:i:s', strtotime($entradaR))
        : '—';
    $salida_despues = !empty($salidaR)
        ? date('d/m/Y H:i:s', strtotime($salidaR))
        : '—';

    $tiempo_texto = $tiempo_totalR ?: '00:00:00';

    $observacion = sprintf(
        '%s [Registro modificado por ADMIN (%s)] Entrada: %s → %s | Salida: %s → %s | Tiempo: %s',
        $timestamp_formateado,
        $nombre_admin,
        $entrada_antes,
        $entrada_despues,
        $salida_antes,
        $salida_despues,
        $tiempo_texto
    );

    $wpdb->insert("{$wpdb->prefix}observaciones", [
        'user_id'          => $trabajador_id,
        'cif_usuario'      => $cif_trabajador,
        'fecha_observacion'=> $fecha_observacion,
        'observaciones'    => $observacion
    ]);

    $mensaje_exito = '✅ Registro actualizado correctamente.';
}

// ---------------------------------------------------
// INTERFAZ ADMIN
// ---------------------------------------------------
?>
<div class="wrap">
    <h1>Revisión de Fichajes (Admin)</h1>

    <?php if ($mensaje_exito): ?>
        <div class="notice notice-success is-dismissible">
            <p><?php echo esc_html($mensaje_exito); ?></p>
        </div>
    <?php endif; ?>

    <form method="post" class="pure-form pure-form-stacked" style="margin-bottom:15px;">
        <fieldset>
            <div style="display:flex; flex-wrap:wrap; gap:10px; align-items:center;">

                <div>
                    <label>Empresa:</label>
                    <select name="empresa_id" onchange="this.form.submit()">
                        <?php foreach ($empresas as $e): 
                            $label = ($e->razon_social ?: $e->user_cif) . ' (' . $e->user_cif . ')';
                        ?>
                            <option value="<?php echo esc_attr($e->id); ?>" <?php selected($e->id, $empresa_id_activa); ?>>
                                <?php echo esc_html($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <label>Desde:</label>
                    <input type="date" name="fecha_inicio" value="<?php echo esc_attr($fecha_inicio); ?>" required>
                </div>

                <div>
                    <label>Hasta:</label>
                    <input type="date" name="fecha_fin" value="<?php echo esc_attr($fecha_fin); ?>" required>
                </div>

                <div>
                    <label>Trabajador:</label>
                    <select name="trabajador_id">
                        <option value="">-- Selecciona --</option>
                        <?php foreach ($trabajadores as $t): 
                            $dni = trim((string)($t->user_dni ?? ''));
                            $dni_mostrar = ($dni !== '') ? $dni : 'sin DNI';
                            $nombre_mostrar = $t->display_name
                                . ($t->user_rol === 'Gestor' ? ' (Gestor)' : '')
                                . ' (' . $dni_mostrar . ')';
                        ?>
                            <option value="<?php echo esc_attr($t->ID); ?>" <?php selected($trabajador_id, $t->ID); ?>>
                                <?php echo esc_html($nombre_mostrar); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <button type="submit" class="button button-primary">Consultar</button>
                </div>

            </div>
        </fieldset>
    </form>

    <div style="margin-bottom:15px;">
        <form method="post" style="display:inline;">
            <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id_activa); ?>">
            <input type="hidden" name="fecha_inicio" value="<?php echo esc_attr($fecha_inicio); ?>">
            <input type="hidden" name="fecha_fin" value="<?php echo esc_attr($fecha_fin); ?>">
            <input type="hidden" name="accion_listar_ausentes" value="1">
            <button type="submit" class="button">🚫 Trabajadores sin fichar hoy</button>
        </form>
    </div>

    <?php
    // AUSENTES HOY
    if (!empty($_POST['accion_listar_ausentes']) && $user_cif) {
        $hoy = current_time('Y-m-d');

        $trabajadores_cif = $wpdb->get_results($wpdb->prepare(
            "SELECT ID, display_name, user_dni FROM {$wpdb->prefix}users 
             WHERE user_cif = %s 
               AND user_estado = 'Activado' 
               AND user_rol IN ('Trabajador', 'Gestor')
             ORDER BY display_name ASC",
            $user_cif
        ));

        $ids_con_fichaje = $wpdb->get_col($wpdb->prepare(
            "SELECT DISTINCT user_id FROM {$wpdb->prefix}registro_horario 
             WHERE DATE(entradaR) = %s AND cif_usuario = %s",
            $hoy, $user_cif
        ));

        $ausentes = array_filter($trabajadores_cif, function($t) use ($ids_con_fichaje) {
            return !in_array($t->ID, $ids_con_fichaje);
        });

        echo "<h3>👥 Trabajadores sin fichaje hoy ($hoy)</h3>";
        if ($ausentes) {
            echo "<ul style='background:#fff3f3; padding:15px; border-radius:5px;'>";
            foreach ($ausentes as $trabajador) {
                $dni = trim((string)($trabajador->user_dni ?? ''));
                $dni_mostrar = $dni !== '' ? $dni : 'sin DNI';
                echo "<li><strong>" . esc_html($trabajador->display_name . ' (' . $dni_mostrar . ')') . "</strong></li>";
            }
            echo "</ul>";
        } else {
            echo "<p style='color:green; font-weight:bold;'>Todos han fichado hoy. ✅</p>";
        }
    }

    // SI HAY TRABAJADOR SELECCIONADO, MOSTRAR DETALLE
    if (!empty($trabajador_id) && !empty($fecha_inicio) && !empty($fecha_fin)):

        // Total de horas en rango
        $total_segundos_global = 0;
        $registros_global = $wpdb->get_results($wpdb->prepare(
            "SELECT entradaR, salidaR 
             FROM {$wpdb->prefix}registro_horario
             WHERE user_id = %d 
               AND DATE(entradaR) BETWEEN %s AND %s",
            $trabajador_id, $fecha_inicio, $fecha_fin
        ));

        foreach ($registros_global as $r) {
            if ($r->entradaR && $r->salidaR) {
                $inicio = strtotime($r->entradaR);
                $fin    = strtotime($r->salidaR);
                if ($fin > $inicio) {
                    $total_segundos_global += ($fin - $inicio);
                }
            }
        }

        $total_tiempo = sprintf('%02d:%02d:%02d',
            floor($total_segundos_global / 3600),
            floor(($total_segundos_global % 3600) / 60),
            $total_segundos_global % 60
        );

        $inicio_anio = date('Y-01-01');

        $total_desde_enero = $wpdb->get_var($wpdb->prepare(
            "SELECT SEC_TO_TIME(SUM(TIME_TO_SEC(tiempo_totalR)))
             FROM {$wpdb->prefix}registro_horario
             WHERE user_id = %d AND entradaR >= %s",
            $trabajador_id, $inicio_anio
        ));

        $segundos_desde_enero = $wpdb->get_var($wpdb->prepare(
            "SELECT SUM(TIME_TO_SEC(tiempo_totalR))
             FROM {$wpdb->prefix}registro_horario
             WHERE user_id = %d AND entradaR >= %s",
            $trabajador_id, $inicio_anio
        ));

        $dias_transcurridos = (new DateTime())->diff(new DateTime($inicio_anio))->days + 1;
        $dias_totales       = (new DateTime(date('Y') . '-12-31'))->format('z') + 1;

        $estimacion_texto = '';
        if ($segundos_desde_enero && $dias_transcurridos > 0) {
            $media_diaria     = $segundos_desde_enero / $dias_transcurridos;
            $estimacion_total = $media_diaria * $dias_totales;
            $horas_estimadas  = floor($estimacion_total / 3600);
            $estimacion_texto = ' · Estimado: ' . $horas_estimadas . 'h';
        }
        ?>

        <div style="text-align:center; font-size:18px; font-weight:bold; margin-top:10px;">
            <p>Total en el periodo: <?php echo $total_tiempo ? esc_html($total_tiempo) : '00:00:00'; ?></p>
            <p style="font-size:13px; color:#888; margin-top:4px;">
                Desde el 1 de enero: <?php echo ($total_desde_enero ?: '00:00:00') . $estimacion_texto; ?>
            </p>
        </div>

        <div style="margin:10px 0;">
            <?php
            // ¿Hay registro abierto?
            $abierta_id = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM mnkt_registro_horario
                 WHERE user_id = %d AND cif_usuario = %s AND salida IS NULL
                 ORDER BY entrada DESC LIMIT 1",
                $trabajador_id, $user_cif
            ));
            $es_salida   = !empty($abierta_id);
            $label_ticar = $es_salida ? 'Ticar salida' : 'Ticar entrada';
            ?>
            <form method="post" style="display:inline-block; margin-right:10px;">
                <?php wp_nonce_field('ticar_nonce', 'ticar_nonce_field'); ?>
                <input type="hidden" name="accion_ticar" value="1">
                <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id_activa); ?>">
                <input type="hidden" name="trabajador_id" value="<?php echo esc_attr($trabajador_id); ?>">
                <input type="hidden" name="fecha_inicio" value="<?php echo esc_attr($fecha_inicio); ?>">
                <input type="hidden" name="fecha_fin" value="<?php echo esc_attr($fecha_fin); ?>">
                <button type="submit" class="button" onclick="return confirm('¿Confirmas <?php echo $es_salida ? 'ticar la SALIDA' : 'ticar la ENTRADA'; ?> para este trabajador?');">
                    ✔️ <?php echo esc_html($label_ticar); ?> (admin)
                </button>
            </form>
        </div>

       <h2>Registros de horario</h2>

<div class="rh-contenedor">
    <table class="pure-table pure-table-bordered" style="width:100%; max-width:100%;">
        <thead>
            <tr>
                <th>Entrada</th>
                <th>Salida</th>
                <th>Tiempo</th>
                <th>Total día</th>
                <th>Guardar</th>
            </tr>
        </thead>
        <tbody>
        <?php
        $registros = $wpdb->get_results($wpdb->prepare(
            "SELECT id, entradaR, salidaR 
             FROM mnkt_registro_horario
             WHERE user_id = %d 
               AND DATE(entradaR) BETWEEN %s AND %s
             ORDER BY entradaR DESC",
            $trabajador_id, $fecha_inicio, $fecha_fin
        ));

        if ($registros) {
            $registros_por_dia = [];

            // Agrupar por fecha
            foreach ($registros as $registro) {
                $fecha = date('Y-m-d', strtotime($registro->entradaR));
                $registros_por_dia[$fecha][] = $registro;
            }

            krsort($registros_por_dia);
            $alternar_color = false;

            foreach ($registros_por_dia as $fecha_dia => $registros_dia):
                $rowspan = count($registros_dia);
                $alternar_color = !$alternar_color;
                $fondo_color = $alternar_color ? '#f8f8f8' : '#ffffff';

                // Total del día
                $total_segundos = 0;
                foreach ($registros_dia as $r) {
                    if ($r->entradaR && $r->salidaR) {
                        $inicio = strtotime($r->entradaR);
                        $fin    = strtotime($r->salidaR);
                        if ($fin > $inicio) {
                            $total_segundos += ($fin - $inicio);
                        }
                    }
                }

                $total_dia = sprintf('%02d:%02d:%02d',
                    floor($total_segundos / 3600),
                    floor(($total_segundos % 3600) / 60),
                    $total_segundos % 60
                );

                foreach ($registros_dia as $i => $registro):
                    // Tiempo exacto de este registro
                    $tiempo_exacto = '00:00:00';
                    if ($registro->entradaR && $registro->salidaR) {
                        $inicio = strtotime($registro->entradaR);
                        $fin    = strtotime($registro->salidaR);
                        if ($fin > $inicio) {
                            $diff = $fin - $inicio;
                            $tiempo_exacto = sprintf('%02d:%02d:%02d',
                                floor($diff / 3600),
                                floor(($diff % 3600) / 60),
                                $diff % 60
                            );
                        }
                    }

                    $id = $registro->id;
        ?>
            <tr id="registro-id-<?php echo $id; ?>" style="background-color: <?php echo $fondo_color; ?>;">
                <td>
                    <form method="post" style="display:flex; gap:5px; align-items:center;">
                        <?php wp_nonce_field('editar_fichaje_nonce', 'editar_fichaje_nonce_field'); ?>
                        <input type="datetime-local"
                            name="entradaR"
                            step="1"
                            value="<?php echo esc_attr(date('Y-m-d\TH:i:s', strtotime($registro->entradaR))); ?>"
                            required
                            style="width:100%;">
                </td>

                <td>
                        <?php
                        $salida_valida = !empty($registro->salidaR);
                        $salida_default = $salida_valida
                            ? date('Y-m-d\TH:i:s', strtotime($registro->salidaR))
                            : date('Y-m-d\TH:i:s', strtotime($registro->entradaR));
                        ?>

                        <?php if (!$salida_valida): ?>
                            <span style="color:#cc0000; font-size:11px;">Falta salida</span><br>
                        <?php endif; ?>

                        <input type="datetime-local"
                            name="salidaR"
                               step="1"
 
                            value="<?php echo esc_attr($salida_default); ?>"
                            style="width:100%; margin-top:5px;">
                </td>

                <td>
                    <span><?php echo esc_html($tiempo_exacto); ?></span>
                </td>

                <?php if ($i === 0): ?>
                    <?php
                    list($h, $m, $s) = explode(':', $total_dia);
                    $total_horas = (int)$h + ((int)$m / 60) + ((int)$s / 3600);
                    $estilo_total_dia = 'text-align:center;';
                    if ($total_horas > 12) {
                        $estilo_total_dia .= ' background-color:#ffcccc; color:#000;';
                    } elseif ($total_horas > 8) {
                        $estilo_total_dia .= ' color:#cc0000;';
                    }
                    ?>
                    <td rowspan="<?php echo $rowspan; ?>" style="<?php echo esc_attr($estilo_total_dia); ?>">
                        <strong><?php echo esc_html($total_dia); ?></strong><br>
                        <span style="font-size:11px;"><?php echo esc_html(date('d/m/Y', strtotime($fecha_dia))); ?></span>
                    </td>
                <?php endif; ?>

                <td style="text-align:center;">
                        <input type="hidden" name="empresa_id" value="<?php echo esc_attr($empresa_id_activa); ?>">
                        <input type="hidden" name="trabajador_id" value="<?php echo esc_attr($trabajador_id); ?>">
                        <input type="hidden" name="fecha_inicio" value="<?php echo esc_attr($fecha_inicio); ?>">
                        <input type="hidden" name="fecha_fin" value="<?php echo esc_attr($fecha_fin); ?>">
                        <input type="hidden" name="editar_id" value="<?php echo esc_attr($id); ?>">

                        <button type="submit" class="button button-primary">💾</button>
                    </form>
                </td>
            </tr>
        <?php
                endforeach;
            endforeach;
        } else {
            echo "<tr><td colspan='5'>No hay registros en este periodo.</td></tr>";
        }
        ?>
        </tbody>
    </table>
</div>
 

        <?php
        // Observaciones
        $observaciones = $wpdb->get_results($wpdb->prepare(
            "SELECT fecha_observacion, observaciones 
             FROM {$wpdb->prefix}observaciones 
             WHERE user_id = %d 
               AND fecha_observacion BETWEEN %s AND %s 
             ORDER BY fecha_observacion DESC",
            $trabajador_id, $fecha_inicio, $fecha_fin
        ));
        if ($observaciones): ?>
            <h3 style="margin-top:30px;">🗒️ Observaciones en el periodo</h3>
            <ul style="background:#f9f9f9; padding:15px; border-radius:8px; list-style:disc inside;">
                <?php foreach ($observaciones as $obs): ?>
                    <li>
                        <strong><?php echo esc_html(date('d/m/Y', strtotime($obs->fecha_observacion))); ?>:</strong>
                        <?php echo esc_html($obs->observaciones); ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>

    <?php endif; // fin detalle trabajador ?>
</div>

