<?php
// Evitar acceso directo
if (!defined('ABSPATH')) exit;

// Solo admins/gestores con permisos altos
if (!current_user_can('manage_options')) {
    wp_die(__('No tienes permisos para acceder a esta página.'));
}

global $wpdb;

$tabla_eu   = $wpdb->prefix . 'empresas_usuarios';
$tabla_emp  = $wpdb->prefix . 'empresas';
$tabla_user = $wpdb->prefix . 'users';
$tabla_rh   = $wpdb->prefix . 'registro_horario';
$tabla_obs  = $wpdb->prefix . 'observaciones'; // opcional

// Helper hh:mm:ss
if (!function_exists('rh_format_seconds')) {
    function rh_format_seconds($seconds) {
        $seconds = (int)$seconds;
        $h = floor($seconds / 3600);
        $m = floor(($seconds % 3600) / 60);
        $s = $seconds % 60;
        return sprintf('%02d:%02d:%02d', $h, $m, $s);
    }
}

// ====== Form state ======
$gestor_id   = isset($_GET['gestor_id']) ? (int) $_GET['gestor_id'] : 0;
$empresa_id  = isset($_GET['empresa_id']) ? (int) $_GET['empresa_id'] : 0;
$iso_week    = isset($_GET['iso_week']) ? sanitize_text_field($_GET['iso_week']) : ''; // formato YYYY-Www
$action_send = isset($_GET['enviar']) && $_GET['enviar'] === '1';
$action_send_me = isset($_GET['enviar_a_mi']) && $_GET['enviar_a_mi'] === '1';

$mensaje     = '';

// Construir lista de gestores activados (desde empresas_usuarios)
$gestores = $wpdb->get_results("
    SELECT DISTINCT u.ID, u.display_name, u.user_email
    FROM {$tabla_eu} eu
    JOIN {$tabla_user} u ON u.ID = eu.user_id
    WHERE eu.rol_en_empresa = 'Gestor'
      AND eu.estado = 'Activado'
      AND (u.user_estado IS NULL OR u.user_estado = 'Activado')
    ORDER BY u.display_name ASC
");

// Si hay gestor seleccionado, listar sus empresas gestionadas
$empresas_del_gestor = [];
if ($gestor_id) {
    $empresas_del_gestor = $wpdb->get_results($wpdb->prepare("
        SELECT e.id, e.user_cif, e.razon_social
        FROM {$tabla_eu} eu
        JOIN {$tabla_emp} e ON e.id = eu.empresa_id
        WHERE eu.user_id = %d
          AND eu.rol_en_empresa = 'Gestor'
          AND eu.estado = 'Activado'
        ORDER BY e.razon_social ASC
    ", $gestor_id));
}

// Calcular rango de la semana a partir de input ISO (YYYY-Www)
$inicio_semana = $fin_semana = null;
if ($iso_week) {
    // PHP no parsea input week directo, lo hacemos a mano:
    // $iso_week ej. "2025-W38" → Lunes de esa semana ISO
    if (preg_match('/^(\d{4})-W(\d{2})$/', $iso_week, $m)) {
        $year = (int)$m[1]; $week = (int)$m[2];
        $dt = new DateTime();
        // Set ISO week → lunes
        $dt->setISODate($year, $week, 1)->setTime(0,0,0);
        // Ajustar a zona WP
        try {
            $tz = wp_timezone();
            $dt->setTimezone($tz);
        } catch (\Throwable $e) {}
        $inicio_semana = $dt->format('Y-m-d H:i:s');
        $fin_semana    = $dt->modify('+7 days')->format('Y-m-d H:i:s');
    }
}

// ====== Envío manual ======
if ($action_send && $gestor_id && $empresa_id && $inicio_semana && $fin_semana && check_admin_referer('reenviar_resumen_semana')) {
    $ok = enviar_resumen_manual_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana);
    if ($ok === true) {
        $mensaje = "<div class='notice notice-success'><p>✅ Resumen enviado correctamente.</p></div>";
    } else {
        $mensaje = "<div class='notice notice-error'><p>❌ Error enviando: " . esc_html($ok) . "</p></div>";
    }
}

// ====== Envío a mi email ======
if ($action_send_me && $gestor_id && $empresa_id && $inicio_semana && $fin_semana && check_admin_referer('reenviar_resumen_semana')) {
    $build = build_resumen_html_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana);
    if (is_string($build)) {
        // Devuelve mensaje de error
        $mensaje = "<div class='notice notice-error'><p>❌ Error construyendo resumen: " . esc_html($build) . "</p></div>";
    } else {
        // Enviar a info@educasede.com
        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $asunto  = $build['asunto'] . ' (copia)';
        $ok_mail = wp_mail('info@educasede.com', $asunto, $build['html'], $headers);
        if ($ok_mail) {
            $mensaje = "<div class='notice notice-success'><p>✅ Resumen enviado a info@educasede.com.</p></div>";
        } else {
            $mensaje = "<div class='notice notice-error'><p>❌ wp_mail devolvió false (¿SMTP?).</p></div>";
        }
    }
}


// Pintar UI
echo $mensaje;
?>
<div class="card" style="background:#fff;border:1px solid #ccd0d4;padding:16px;max-width:920px;">
    <h2 style="margin-top:0;">📧 Reenviar resumen semanal</h2>
    <p>Selecciona gestor, su empresa y la semana (ISO) para reenviar el informe.</p>

    <form method="get" action="">
        <input type="hidden" name="page" value="control-horario" />
        <input type="hidden" name="tab" value="resumen" />

        <table class="form-table" role="presentation">
            <tr>
                <th><label for="gestor_id">Gestor</label></th>
                <td>
                    <select name="gestor_id" id="gestor_id" onchange="this.form.submit()">
                        <option value="0">— Selecciona gestor —</option>
                        <?php foreach ($gestores as $g): ?>
                            <option value="<?php echo (int)$g->ID; ?>" <?php selected($gestor_id, (int)$g->ID); ?>>
                                <?php echo esc_html($g->display_name . ' — ' . $g->user_email); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <p class="description">Al seleccionar, se cargarán sus empresas.</p>
                </td>
            </tr>

            <tr>
                <th><label for="empresa_id">Empresa</label></th>
                <td>
                    <select name="empresa_id" id="empresa_id" <?php disabled(!$gestor_id); ?>>
                        <option value="0">— Selecciona empresa —</option>
                        <?php foreach ($empresas_del_gestor as $e): ?>
                            <option value="<?php echo (int)$e->id; ?>" <?php selected($empresa_id, (int)$e->id); ?>>
                                <?php
                                    $label = $e->razon_social ? "{$e->razon_social} ({$e->user_cif})" : $e->user_cif;
                                    echo esc_html($label);
                                ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <?php if (!$gestor_id): ?>
                        <p class="description">Selecciona antes un gestor.</p>
                    <?php endif; ?>
                </td>
            </tr>

            <tr>
                <th><label for="iso_week">Semana</label></th>
                <td>
                    <input type="week" name="iso_week" id="iso_week" value="<?php echo esc_attr($iso_week); ?>" required />
                    <p class="description">Formato ISO (ej. 2025-W38). Se toma de lunes 00:00 a lunes siguiente 00:00.</p>
                </td>
            </tr>
        </table>

        <?php wp_nonce_field('reenviar_resumen_semana'); ?>

        
    <button type="submit" class="button button-secondary">Cargar selección</button>
    <button type="submit" name="enviar" value="1" class="button button-primary" <?php disabled(!($gestor_id && $empresa_id && $iso_week)); ?>>
        Enviar ahora
    </button>
    <!-- 👇 NUEVO botón para enviar a tu email -->
    <button type="submit" name="enviar_a_mi" value="1" class="button">
        Enviar a mi email (info@educasede.com)
    </button>
</p>

        </p>
    </form>
<?php if ($gestor_id && $empresa_id && $inicio_semana && $fin_semana): ?>
    <?php
    $build = build_resumen_html_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana);
    if (!is_string($build)): // ok
    ?>
        <div class="card" style="background:#fff;border:1px solid #ccd0d4;padding:16px;max-width:920px;margin-top:12px;">
            <h3 style="margin:0 0 10px;">🔎 Previsualización</h3>
            <p><strong>Gestor:</strong> <?php echo esc_html($build['gestor']->display_name . ' — ' . $build['gestor']->user_email); ?></p>
            <p><strong>Empresa:</strong> 
                <?php 
                    $lab = $build['empresa']->razon_social ? "{$build['empresa']->razon_social} ({$build['empresa']->user_cif})" : $build['empresa']->user_cif;
                    echo esc_html($lab);
                ?>
            </p>
            <p><strong>Rango:</strong> <?php echo esc_html($build['rango_txt']); ?></p>
            <hr>
            <div><?php echo $build['html']; // HTML completo del email ?></div>
        </div>
    <?php else: ?>
        <div class="notice notice-error"><p>❌ Error en previsualización: <?php echo esc_html($build); ?></p></div>
    <?php endif; ?>
<?php endif; ?>

    <?php if ($inicio_semana && $fin_semana && $empresa_id): ?>
        <p><strong>Rango calculado:</strong> <?php
            echo esc_html( date_i18n('d/m/Y H:i', strtotime($inicio_semana)) . ' → ' . date_i18n('d/m/Y H:i', strtotime($fin_semana)) );
        ?></p>
    <?php endif; ?>
</div>

<?php


/**
 * Construye el HTML del resumen (y metadatos) para un gestor, empresa y rango.
 * Devuelve array con: html, asunto, gestor (obj WP_User), empresa (row), rango_txt
 * o string con mensaje de error.
 */
function build_resumen_html_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana) {
    global $wpdb;

    $tabla_eu   = $wpdb->prefix . 'empresas_usuarios';
    $tabla_emp  = $wpdb->prefix . 'empresas';
    $tabla_user = $wpdb->prefix . 'users';
    $tabla_rh   = $wpdb->prefix . 'registro_horario';
    $tabla_obs  = $wpdb->prefix . 'observaciones';

    // Validaciones
    $gestor = get_userdata($gestor_id);
    if (!$gestor) return 'Gestor no encontrado';

    $ok = (int)$wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$tabla_eu}
        WHERE user_id = %d AND empresa_id = %d AND rol_en_empresa='Gestor' AND estado='Activado'
    ", $gestor_id, $empresa_id));
    if (!$ok) return 'Ese gestor no gestiona esa empresa (o está desactivado).';

    $empresa = $wpdb->get_row($wpdb->prepare("
        SELECT id, user_cif, razon_social FROM {$tabla_emp} WHERE id = %d
    ", $empresa_id));
    if (!$empresa) return 'Empresa no encontrada';

    // Usuarios activados
    $trabajadores = $wpdb->get_results($wpdb->prepare("
        SELECT u.ID, u.display_name
        FROM {$tabla_eu} eu
        JOIN {$tabla_user} u ON u.ID = eu.user_id
        WHERE eu.empresa_id = %d
          AND eu.estado = 'Activado'
          AND eu.rol_en_empresa IN ('Trabajador','Gestor','Técnico')
          AND (u.user_estado IS NULL OR u.user_estado = 'Activado')
        ORDER BY u.display_name ASC
    ", $empresa_id));

    // Días semana
    $dias_semana = [
        'Monday'    => 'Lunes',
        'Tuesday'   => 'Martes',
        'Wednesday' => 'Miércoles',
        'Thursday'  => 'Jueves',
        'Friday'    => 'Viernes',
        'Saturday'  => 'Sábado',
        'Sunday'    => 'Domingo'
    ];

    // Cuerpo tabla + obs
    $resumenHTML = '';
    $observacionesHTML = '';

    foreach ($trabajadores as $t) {
        $user_id = (int)$t->ID;
        $nombre  = esc_html($t->display_name);

        $registros = $wpdb->get_results($wpdb->prepare("
            SELECT DATE(COALESCE(entradaR, entrada)) AS dia,
                   SUM(TIME_TO_SEC(COALESCE(tiempo_totalR, tiempo_total))) AS total_seg_dia
            FROM {$tabla_rh}
            WHERE user_id = %d
              AND (
                    cif_id = %d
                 OR (cif_id IS NULL AND cif_usuario = %s)
              )
              AND COALESCE(entradaR, entrada) >= %s
              AND COALESCE(entradaR, entrada) < %s
              AND COALESCE(tiempo_totalR, tiempo_total) IS NOT NULL
            GROUP BY DATE(COALESCE(entradaR, entrada))
        ", $user_id, $empresa_id, $empresa->user_cif, $inicio_semana, $fin_semana));

        $tiempo_por_dia   = [];
        $tiempo_total_seg = 0;

        foreach ($registros as $r) {
            $seg = (int)$r->total_seg_dia;
            $tiempo_total_seg += $seg;
            $dia_php = date('l', strtotime($r->dia));
            $tiempo_por_dia[$dia_php] = $seg;
        }

        $resumenHTML .= "<tr style='border-bottom:1px solid #ccc;'>
            <td style='padding:10px;text-align:center;'>{$nombre}</td>";

        foreach (array_keys($dias_semana) as $diaEN) {
            $dur = isset($tiempo_por_dia[$diaEN]) ? rh_format_seconds($tiempo_por_dia[$diaEN]) : '&mdash;';
            $resumenHTML .= "<td style='padding:10px;text-align:center;'>{$dur}</td>";
        }

        $total_txt = $tiempo_total_seg > 0 ? rh_format_seconds($tiempo_total_seg) : '&mdash;';
        $resumenHTML .= "<td style='padding:10px;text-align:center;font-weight:bold;'>{$total_txt}</td></tr>";

        // Observaciones si la tabla existe
        $tabla_obs_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $tabla_obs));
        if ($tabla_obs_exists) {
            $obs = $wpdb->get_results($wpdb->prepare("
                SELECT fecha_observacion, observaciones
                FROM {$tabla_obs}
                WHERE user_id = %d
                  AND fecha_observacion BETWEEN %s AND %s
                ORDER BY fecha_observacion ASC
            ", $user_id, substr($inicio_semana,0,10), substr($fin_semana,0,10)));

            if ($obs) {
                $observacionesHTML .= "<h4 style='margin-top:30px;'>Observaciones de {$nombre}</h4><ul style='padding-left:20px;'>";
                foreach ($obs as $o) {
                    $fecha = date('d/m/Y', strtotime($o->fecha_observacion));
                    $observacionesHTML .= "<li><strong>{$fecha}:</strong> " . esc_html($o->observaciones) . "</li>";
                }
                $observacionesHTML .= '</ul>';
            }
        }
    }

    // Tabla completa
    $thead = "<tr><th style='padding:10px;text-align:center;'>Nombre</th>";
    foreach ($dias_semana as $dES) {
        $thead .= "<th style='padding:10px;text-align:center;'>{$dES}</th>";
    }
    $thead .= "<th style='padding:10px;text-align:center;'>Total Semana</th></tr>";

    $tablaResumen = "<table style='width:100%; border-collapse: collapse; margin-top: 20px;'>
        <thead style='background-color:#0073aa; color:#fff;'>{$thead}</thead>
        <tbody>{$resumenHTML}</tbody>
    </table>";

    // Plantilla
    $template_path = WP_PLUGIN_DIR . '/control-horario-trabajadores/templates/resumen_semanal_empresa.html';
    if (!file_exists($template_path)) return 'No se encontró la plantilla de email.';
    $html_tpl = file_get_contents($template_path);

    // Rango bonito
    $rango_txt = date_i18n('d/m/Y', strtotime($inicio_semana)) . ' al ' . date_i18n('d/m/Y', strtotime('-1 day', strtotime($fin_semana)));

    // Reemplazos
    $html = str_replace(
        ['${nombreGestor}', '${cifEmpresa}', '${rangoSemana}', '${tablaResumenHTML}', '${tablaObservacionesHTML}', '${nombreEmpresa}'],
        [$gestor->display_name, $empresa->user_cif, $rango_txt, $tablaResumen, $observacionesHTML, ($empresa->razon_social ?: '')],
        $html_tpl
    );

    $asunto = "📊 Resumen semanal de actividad - {$empresa->user_cif}" . ($empresa->razon_social ? " ({$empresa->razon_social})" : '');

    return [
        'html'       => $html,
        'asunto'     => $asunto,
        'gestor'     => $gestor,
        'empresa'    => $empresa,
        'rango_txt'  => $rango_txt,
    ];
}




/**
 * Envía el resumen SOLO para un gestor, una empresa y un rango concreto
 * Devuelve true si ok, o string de error si falla.
 */



function enviar_resumen_manual_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana) {
    $build = build_resumen_html_gestor_empresa_rango($gestor_id, $empresa_id, $inicio_semana, $fin_semana);
    if (is_string($build)) {
        return $build; // error
    }
    $headers = ['Content-Type: text/html; charset=UTF-8'];
    $ok_mail = wp_mail($build['gestor']->user_email, $build['asunto'], $build['html'], $headers);
    return $ok_mail ? true : 'wp_mail devolvió false (¿SMTP?).';
}

