<?php
// Evitar acceso directo
if (!defined('ABSPATH')) {
    exit;
}

global $wpdb;

// Solo admin
if (!current_user_can('manage_options')) {
    echo '<div class="notice notice-error"><p>❌ No tienes permisos para acceder a esta página.</p></div>';
    return;
}

// --------------------------------------------------
// Utilidades internas
// --------------------------------------------------

/**
 * Obtener ranking de puntos por empresa y periodo.
 */
function ch_ch_get_ranking_puntos($user_cif, $fecha_inicio, $fecha_fin) {
    global $wpdb;

    $tabla_registro = $wpdb->prefix . 'registro_horario';
    $tabla_users    = $wpdb->prefix . 'users';

    // Sumamos puntos por usuario en el rango de fechas
    $sql = $wpdb->prepare("
        SELECT 
            u.ID,
            u.display_name,
            u.user_email,
            u.user_rol,
            COALESCE(SUM(r.puntos), 0) AS total_puntos,
            COUNT(r.id) AS registros
        FROM $tabla_registro r
        JOIN $tabla_users u ON u.ID = r.user_id
        WHERE r.cif_usuario = %s
          AND r.entrada BETWEEN %s AND %s
          AND u.user_estado = 'Activado'
          AND u.user_rol IN ('Trabajador', 'Gestor')
        GROUP BY u.ID
        HAVING total_puntos <> 0
        ORDER BY total_puntos DESC, u.display_name ASC
    ", $user_cif, $fecha_inicio . ' 00:00:00', $fecha_fin . ' 23:59:59');

    $result = $wpdb->get_results($sql);

    return $result ?: [];
}

/**
 * Construir tabla HTML del ranking.
 */
function ch_ch_build_ranking_table_html($ranking) {

    if (empty($ranking)) {
        return '<p style="text-align:center; color:#777;">No hay puntos registrados en este periodo.</p>';
    }

    $html  = '<table style="width:100%; border-collapse:collapse; font-size:14px;">';
    $html .= '<thead>';
    $html .= '<tr>';
    $html .= '<th style="text-align:left; padding:8px; border-bottom:1px solid #ddd;">Pos.</th>';
    $html .= '<th style="text-align:left; padding:8px; border-bottom:1px solid #ddd;">Nombre</th>';
    $html .= '<th style="text-align:center; padding:8px; border-bottom:1px solid #ddd;">Rol</th>';
    $html .= '<th style="text-align:right; padding:8px; border-bottom:1px solid #ddd;">Puntos</th>';
    $html .= '<th style="text-align:right; padding:8px; border-bottom:1px solid #ddd;">Registros</th>';
    $html .= '</tr>';
    $html .= '</thead><tbody>';

    $pos = 1;
    foreach ($ranking as $row) {
        $medalla = '';
        if ($pos === 1)      $medalla = '🥇';
        elseif ($pos === 2)  $medalla = '🥈';
        elseif ($pos === 3)  $medalla = '🥉';

        $rol_label = ($row->user_rol === 'Gestor') ? 'Gestor' : 'Trabajador';

        $html .= '<tr>';
        $html .= '<td style="padding:6px 8px; border-bottom:1px solid #f0f0f0;">' . esc_html($pos) . ' ' . $medalla . '</td>';
        $html .= '<td style="padding:6px 8px; border-bottom:1px solid #f0f0f0;">' . esc_html($row->display_name) . '</td>';
        $html .= '<td style="padding:6px 8px; border-bottom:1px solid #f0f0f0; text-align:center; color:#555;">' . esc_html($rol_label) . '</td>';
        $html .= '<td style="padding:6px 8px; border-bottom:1px solid #f0f0f0; text-align:right; font-weight:600;">' . esc_html($row->total_puntos) . '</td>';
        $html .= '<td style="padding:6px 8px; border-bottom:1px solid #f0f0f0; text-align:right; color:#777;">' . esc_html($row->registros) . '</td>';
        $html .= '</tr>';

        $pos++;
    }

    $html .= '</tbody></table>';
    return $html;
}

/**
 * Construir el HTML final del email usando la plantilla.
 */
        function ch_ch_build_ranking_email_html($empresa_nombre, $periodo_texto, $tablaRankingHTML) {
    // 1) Directorio base del plugin (subimos dos niveles desde /includes/...)
    $plugin_root = dirname( dirname( plugin_dir_path(__FILE__) ) );
    // Resultado esperado:
    // /home/admin/domains/educasede.com/public_html/app/wp-content/plugins/control-horario-trabajadores

    // 2) Ruta correcta a la plantilla
    $template_path = $plugin_root . '/templates/rankings.html';

    if (file_exists($template_path)) {
        $html = file_get_contents($template_path);
    } else {
        // Déjalo mientras pruebas; luego lo puedes quitar
        $html = '<html><body><h2>Plantilla no encontrada en: ' . htmlspecialchars($template_path) . '</h2></body></html>';
    }

    $reemplazos = [
        '${nombreEmpresa}'    => $empresa_nombre,
        '${periodoTexto}'     => $periodo_texto,
        '${tablaRankingHTML}' => $tablaRankingHTML,
    ];

    return strtr($html, $reemplazos);
}




// --------------------------------------------------
// Carga de empresas
// --------------------------------------------------

$empresas = $wpdb->get_results("
    SELECT id, user_cif, razon_social
    FROM {$wpdb->prefix}empresas
    ORDER BY razon_social ASC, user_cif ASC
");

if (!$empresas) {
    echo '<div class="notice notice-warning"><p>No hay empresas configuradas.</p></div>';
    return;
}

// Valores iniciales de formulario
$hoy          = current_time('Y-m-d');
$anio_actual  = (int) date('Y');

$empresa_id   = isset($_POST['empresa_id']) ? intval($_POST['empresa_id']) : (int)$empresas[0]->id;
$tipo_periodo = isset($_POST['tipo_periodo']) ? sanitize_text_field($_POST['tipo_periodo']) : 'mes';
$mes          = isset($_POST['mes']) ? intval($_POST['mes']) : intval(date('n'));
$trimestre    = isset($_POST['trimestre']) ? intval($_POST['trimestre']) : 1;
$anio         = isset($_POST['anio']) ? intval($_POST['anio']) : $anio_actual;

$accion       = isset($_POST['accion']) ? sanitize_text_field($_POST['accion']) : 'preview';

$empresa_activa = null;
foreach ($empresas as $e) {
    if ((int)$e->id === $empresa_id) {
        $empresa_activa = $e;
        break;
    }
}
if (!$empresa_activa) {
    $empresa_activa = $empresas[0];
    $empresa_id     = (int)$empresa_activa->id;
}

$user_cif      = $empresa_activa->user_cif;
$empresa_nombre = $empresa_activa->razon_social ?: $empresa_activa->user_cif;

// --------------------------------------------------
// Cálculo periodo
// --------------------------------------------------

$fecha_inicio = $hoy;
$fecha_fin    = $hoy;
$periodo_texto = '';

if ($tipo_periodo === 'mes') {
    $fecha_inicio = sprintf('%04d-%02d-01', $anio, $mes);
    $fecha_fin    = date('Y-m-t', strtotime($fecha_inicio)); // último día del mes
    $periodo_texto = sprintf('Mes %02d/%04d', $mes, $anio);
} elseif ($tipo_periodo === 'trimestre') {
    $mes_inicio = ($trimestre - 1) * 3 + 1;
    $fecha_inicio = sprintf('%04d-%02d-01', $anio, $mes_inicio);
    $mes_fin = $mes_inicio + 2;
    $fecha_fin = date('Y-m-t', strtotime(sprintf('%04d-%02d-01', $anio, $mes_fin)));
    $periodo_texto = sprintf('Trimestre %d de %04d', $trimestre, $anio);
} else { // anio
    $fecha_inicio = sprintf('%04d-01-01', $anio);
    $fecha_fin    = sprintf('%04d-12-31', $anio);
    $periodo_texto = sprintf('Año %04d', $anio);
}

// --------------------------------------------------
// Si hay que generar ranking / enviar
// --------------------------------------------------

$ranking           = [];
$tabla_ranking_html = '';
$email_html        = '';
$mensaje_estado    = '';

if (!empty($_POST)) {
    $ranking = ch_ch_get_ranking_puntos($user_cif, $fecha_inicio, $fecha_fin);
    $tabla_ranking_html = ch_ch_build_ranking_table_html($ranking);
    $email_html = ch_ch_build_ranking_email_html($empresa_nombre, $periodo_texto, $tabla_ranking_html);

    if ($accion === 'test' && !empty($email_html)) {
        // Enviar prueba a info@educasede.com
        $to = 'info@educasede.com';
        $subject = 'PRUEBA · Ranking de puntos · ' . $empresa_nombre . ' · ' . $periodo_texto;
        $headers = ['Content-Type: text/html; charset=UTF-8'];

        if (wp_mail($to, $subject, $email_html, $headers)) {
            $mensaje_estado = '✅ Email de prueba enviado a info@educasede.com.';
        } else {
            $mensaje_estado = '❌ Error al enviar la prueba a info@educasede.com.';
        }

    } elseif ($accion === 'enviar_gestores' && !empty($email_html)) {

        // Obtener emails de gestores de la empresa
        $gestores_emails = $wpdb->get_col($wpdb->prepare(
            "SELECT u.user_email
             FROM {$wpdb->prefix}empresas_usuarios eu
             JOIN {$wpdb->prefix}users u ON u.ID = eu.user_id
             WHERE eu.empresa_id = %d
               AND eu.rol_en_empresa = 'Gestor'
               AND eu.estado = 'Activado'",
            $empresa_id
        ));

        if (!empty($gestores_emails)) {
            $to = implode(',', array_map('sanitize_email', $gestores_emails));
            $subject = 'Ranking de puntos · ' . $empresa_nombre . ' · ' . $periodo_texto;
            $headers = ['Content-Type: text/html; charset=UTF-8'];

            if (wp_mail($to, $subject, $email_html, $headers)) {
                $mensaje_estado = '✅ Ranking enviado a los gestores de la empresa.';
            } else {
                $mensaje_estado = '❌ Error al enviar el ranking a los gestores.';
            }
        } else {
            $mensaje_estado = '⚠️ No se han encontrado gestores activos con email en esta empresa.';
        }
    }
}

// --------------------------------------------------
// INTERFAZ BACKEND
// --------------------------------------------------
?>
<div class="wrap">
    <h1>🏆 Rankings de gamificación</h1>
    <p>Desde aquí puedes generar y enviar el ranking de puntos por empresa y periodo.</p>

    <?php if ($mensaje_estado): ?>
        <div class="notice notice-info is-dismissible">
            <p><?php echo esc_html($mensaje_estado); ?></p>
        </div>
    <?php endif; ?>

    <form method="post" style="margin-bottom:20px; padding:10px 15px; background:#fff; border:1px solid #ddd; border-radius:6px;">
        <table class="form-table">
            <tr>
                <th scope="row"><label for="empresa_id">Empresa</label></th>
                <td>
                    <select name="empresa_id" id="empresa_id">
                        <?php foreach ($empresas as $e): 
                            $label = ($e->razon_social ?: $e->user_cif) . ' (' . $e->user_cif . ')';
                        ?>
                            <option value="<?php echo esc_attr($e->id); ?>" <?php selected($e->id, $empresa_id); ?>>
                                <?php echo esc_html($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>

            <tr>
                <th scope="row">Periodo</th>
                <td>
                    <label>
                        <input type="radio" name="tipo_periodo" value="mes" <?php checked($tipo_periodo, 'mes'); ?>>
                        Mes
                    </label>
                    &nbsp;&nbsp;
                    <label>
                        <input type="radio" name="tipo_periodo" value="trimestre" <?php checked($tipo_periodo, 'trimestre'); ?>>
                        Trimestre
                    </label>
                    &nbsp;&nbsp;
                    <label>
                        <input type="radio" name="tipo_periodo" value="anio" <?php checked($tipo_periodo, 'anio'); ?>>
                        Año
                    </label>

                    <div style="margin-top:8px;">
                        <span id="selector-mes" style="<?php echo ($tipo_periodo === 'mes') ? '' : 'display:none;'; ?>">
                            Mes: 
                            <select name="mes">
                                <?php for ($m = 1; $m <= 12; $m++): ?>
                                    <option value="<?php echo esc_attr($m); ?>" <?php selected($mes, $m); ?>>
                                        <?php echo esc_html($m); ?>
                                    </option>
                                <?php endfor; ?>
                            </select>
                        </span>

                        <span id="selector-trimestre" style="<?php echo ($tipo_periodo === 'trimestre') ? '' : 'display:none;'; ?>">
                            Trimestre:
                            <select name="trimestre">
                                <option value="1" <?php selected($trimestre, 1); ?>>1º</option>
                                <option value="2" <?php selected($trimestre, 2); ?>>2º</option>
                                <option value="3" <?php selected($trimestre, 3); ?>>3º</option>
                                <option value="4" <?php selected($trimestre, 4); ?>>4º</option>
                            </select>
                        </span>

                        <span style="margin-left:10px;">
                            Año:
                            <input type="number" name="anio" value="<?php echo esc_attr($anio); ?>" style="width:80px;">
                        </span>
                    </div>
                </td>
            </tr>
        </table>

        <p class="submit">
            <button type="submit" name="accion" value="preview" class="button button-secondary">👀 Ver ranking</button>
            <button type="submit" name="accion" value="test" class="button">✉️ Enviar prueba a info@educasede.com</button>
            <button type="submit" name="accion" value="enviar_gestores" class="button button-primary">🚀 Enviar a gestores</button>
        </p>
    </form>

    <?php if (!empty($email_html)): ?>
        <h2>Vista previa del email</h2>
        <div style="border:1px solid #ddd; border-radius:6px; overflow:hidden; background:#f3f4f6; padding:10px;">
            <div style="background:white; max-width:800px; margin:auto;">
                <?php echo $email_html; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const radios = document.querySelectorAll('input[name="tipo_periodo"]');
    const selMes = document.getElementById('selector-mes');
    const selTri = document.getElementById('selector-trimestre');

    function actualizarVisibilidad() {
        const tipo = document.querySelector('input[name="tipo_periodo"]:checked').value;
        if (tipo === 'mes') {
            selMes.style.display = '';
            selTri.style.display = 'none';
        } else if (tipo === 'trimestre') {
            selMes.style.display = 'none';
            selTri.style.display = '';
        } else {
            selMes.style.display = 'none';
            selTri.style.display = 'none';
        }
    }

    radios.forEach(r => r.addEventListener('change', actualizarVisibilidad));
    actualizarVisibilidad();
});
</script>
