<?php
if (!defined('ABSPATH')) exit;

global $wpdb;
$mensaje = $mensaje ?? '';
$tabla_usuarios = $wpdb->prefix . 'users';
$tabla_empresas = $wpdb->prefix . 'empresas';
$tabla_m2m      = $wpdb->prefix . 'empresas_usuarios';
$editar_gestor = null;

// Cargar empresas activas
$empresas = $wpdb->get_results("
    SELECT id, user_cif, razon_social
    FROM $tabla_empresas
    WHERE estado_empresa = 'activado'
    ORDER BY razon_social ASC
");

// Helper: asegurar vinculación M2M (idempotente)
function eg_ensure_m2m_link($tabla_m2m, $user_id, $empresa_id) {
    global $wpdb;
    if ($user_id <= 0 || $empresa_id <= 0) return false;

    // Intento 1: si hay UNIQUE(user_id,empresa_id)
    $sql = $wpdb->prepare(
        "INSERT INTO $tabla_m2m (user_id, empresa_id, creado_en)
         VALUES (%d, %d, %s)
         ON DUPLICATE KEY UPDATE empresa_id = empresa_id",
        $user_id, $empresa_id, current_time('mysql')
    );
    $res = $wpdb->query($sql);
    if ($res !== false) return true;

    // Intento 2: fallback sin UNIQUE
    $exists = $wpdb->get_var($wpdb->prepare(
        "SELECT id FROM $tabla_m2m WHERE user_id=%d AND empresa_id=%d LIMIT 1",
        $user_id, $empresa_id
    ));
    if ($exists) return true;

    return (bool) $wpdb->insert(
        $tabla_m2m,
        ['user_id' => $user_id, 'empresa_id' => $empresa_id, 'creado_en' => current_time('mysql')],
        ['%d','%d','%s']
    );
}

// Procesar guardar (crear o editar)
if (isset($_POST['guardar_gestor'])) {
    $user_id           = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
    $user_login        = sanitize_text_field($_POST['user_login']);
    $user_email        = sanitize_email($_POST['user_email']);
    $user_dni          = sanitize_text_field($_POST['user_dni']);
    $user_gamificacion = isset($_POST['user_gamificacion']) ? 1 : 0;

    // Resolver empresa seleccionada y/o CIF manual
    $user_cif    = '';
    $user_cif_id = null;

    $empresa_id_sel = isset($_POST['empresa_id_sel']) ? intval($_POST['empresa_id_sel']) : 0;
    if ($empresa_id_sel > 0) {
        $empresa_sel = $wpdb->get_row($wpdb->prepare(
            "SELECT id, user_cif FROM $tabla_empresas WHERE id = %d",
            $empresa_id_sel
        ));
        if ($empresa_sel) {
            $user_cif    = $empresa_sel->user_cif;
            $user_cif_id = (int)$empresa_sel->id;
        }
    }

    // Compatibilidad: si no se seleccionó empresa, usar el campo manual
    if ($user_cif === '') {
        $user_cif = sanitize_text_field($_POST['user_cif'] ?? '');
    }

    if (empty($user_email) || empty($user_login)) {
        $mensaje = "<div class='error'><p>El correo y el nombre de usuario son obligatorios.</p></div>";
    } else {
        if ($user_id) {
            // ---- Editar ----
            $args = ['ID' => $user_id, 'user_email' => $user_email];
            if (!empty($_POST['user_pass'])) {
                $args['user_pass'] = sanitize_text_field($_POST['user_pass']);
            }
            $result = wp_update_user($args);

            if (!is_wp_error($result)) {
                // Actualiza metacampos en la tabla users (campos custom)
                $wpdb->update(
                    $tabla_usuarios,
                    [
                        'user_dni'          => $user_dni,
                        'user_cif'          => $user_cif,
                        'user_cif_id'       => $user_cif_id,
                        'user_gamificacion' => $user_gamificacion,
                        'user_rol'          => 'Gestor'
                    ],
                    ['ID' => $user_id],
                    ['%s','%s','%d','%d','%s'],
                    ['%d']
                );

                // Vinculación M2M si hay empresa seleccionada
                $m2m_info = '';
                if ($user_cif_id) {
                    $ok_link = eg_ensure_m2m_link($tabla_m2m, $user_id, $user_cif_id);
                    $m2m_info = $ok_link ? ' Vinculación M2M verificada/creada.' : ' (No se pudo crear la vinculación M2M).';
                }

                // Migración progresiva: sincronizar gamificación por user_cif_id (si existe)
                if ($user_cif_id) {
                    $wpdb->query($wpdb->prepare(
                        "UPDATE $tabla_usuarios SET user_gamificacion = %d WHERE user_cif_id = %d",
                        $user_gamificacion,
                        $user_cif_id
                    ));
                }

                $mensaje = "<div class='updated'><p>Gestor actualizado correctamente y gamificación sincronizada.$m2m_info</p></div>";
            } else {
                $mensaje = "<div class='error'><p>Error al actualizar gestor: " . $result->get_error_message() . "</p></div>";
            }
        } else {
            // ---- Crear ----
            $password = sanitize_text_field($_POST['user_pass'] ?? '');
            if (empty($password)) {
                // Password sencilla de 8 chars
                $password = substr(md5(uniqid(rand(), true)), 0, 8);
            }

            $new_user_id = wp_insert_user([
                'user_login' => $user_login,
                'user_email' => $user_email,
                'user_pass'  => $password,
                'role'       => 'gestor'
            ]);

            if (!is_wp_error($new_user_id)) {
                // Campos custom en tabla users
                $wpdb->update(
                    $tabla_usuarios,
                    [
                        'user_dni'          => $user_dni,
                        'user_cif'          => $user_cif,
                        'user_cif_id'       => $user_cif_id,
                        'user_gamificacion' => $user_gamificacion,
                        'user_rol'          => 'Gestor'
                    ],
                    ['ID' => $new_user_id],
                    ['%s','%s','%d','%d','%s'],
                    ['%d']
                );

                // Vinculación M2M si hay empresa seleccionada
                $m2m_info = '';
                if ($user_cif_id) {
                    $ok_link = eg_ensure_m2m_link($tabla_m2m, $new_user_id, $user_cif_id);
                    $m2m_info = $ok_link ? ' Vinculación M2M creada.' : ' (No se pudo crear la vinculación M2M).';
                }

                // Migración progresiva: sincronizar gamificación por user_cif_id (si existe)
                if ($user_cif_id) {
                    $wpdb->query($wpdb->prepare(
                        "UPDATE $tabla_usuarios SET user_gamificacion = %d WHERE user_cif_id = %d",
                        $user_gamificacion,
                        $user_cif_id
                    ));
                }

                // Email bienvenida
                $template_path = plugin_dir_path(dirname(dirname(__FILE__))) . 'templates/email-nuevo-gestor.html';
                if (file_exists($template_path)) {
                    $email_body = file_get_contents($template_path);
                    $email_body = str_replace(
                        ['[user_login]', '[user_email]', '[password]'],
                        [$user_login, $user_email, $password],
                        $email_body
                    );
                    $headers = ['Content-Type: text/html; charset=UTF-8', 'From: E-Digitiall <no-reply@educasede.com>'];
                    wp_mail($user_email, 'Bienvenido a E-Digitiall', $email_body, $headers);
                }

                $mensaje = "<div class='updated'><p>Nuevo gestor creado correctamente.$m2m_info</p></div>";
            } else {
                $mensaje = "<div class='error'><p>Error al crear gestor: " . $new_user_id->get_error_message() . "</p></div>";
            }
        }
    }
}

// Eliminar
if (isset($_GET['eliminar']) && is_numeric($_GET['eliminar'])) {
    wp_delete_user(intval($_GET['eliminar']));
    $mensaje = "<div class='updated'><p>Gestor eliminado correctamente.</p></div>";
}

// Reenviar contraseña
if (isset($_GET['reenviar']) && is_numeric($_GET['reenviar'])) {
    $user = get_userdata(intval($_GET['reenviar']));
    if ($user) {
        retrieve_password($user->user_login);
        $mensaje = "<div class='updated'><p>Email de recuperación enviado a " . esc_html($user->user_email) . "</p></div>";
    }
}

// Cargar datos para edición
if (isset($_GET['editar']) && is_numeric($_GET['editar'])) {
    $editar_gestor = get_userdata(intval($_GET['editar']));
}

// Obtener todos los gestores (desde users con campos custom)
$gestores = $wpdb->get_results("
    SELECT 
        u.ID,
        u.user_login,
        u.user_email,
        u.user_dni,
        u.user_cif,
        u.user_cif_id,
        e.user_cif      AS cif_asociado,
        e.razon_social  AS empresa_asociada,
        e.gamificacion  AS gamif_empresa,
        (
            SELECT GROUP_CONCAT(
                       DISTINCT CONCAT(eu.empresa_id, '||', IFNULL(e2.user_cif,''), '||', IFNULL(e2.razon_social,''))
                       ORDER BY eu.empresa_id SEPARATOR '##'
                   )
            FROM {$tabla_m2m} eu
            LEFT JOIN {$tabla_empresas} e2 ON e2.id = eu.empresa_id
            WHERE eu.user_id = u.ID
        ) AS empresas_info
    FROM {$tabla_usuarios} u
    LEFT JOIN {$tabla_empresas} e
           ON e.id = u.user_cif_id
    WHERE u.user_rol = 'Gestor'
    ORDER BY u.ID ASC
");


// Probar envío de email
if (isset($_POST['test_envio_mail'])) {
    $resultado = wp_mail(
        ['info@educasede.com', 'pablom@elbuenpastor.es'],
        'Test wp_mail con headers',
        'Este es un test simple para verificar el envío de correos desde el plugin.',
        ['Content-Type: text/html; charset=UTF-8', 'From: E-Digitiall <no-reply@educasede.com>']
    );

    echo $resultado
        ? '<div class="updated"><p>✅ Email de prueba enviado correctamente.</p></div>'
        : '<div class="error"><p>❌ Fallo al enviar el email de prueba.</p></div>';
}
?>

<h1>Gestión de Gestores</h1>
<a href="?page=control-horario&tab=gestores&nuevo_gestor=1" class="button button-primary">Añadir Nuevo Gestor</a>

<?php echo $mensaje; ?>

<?php if ($editar_gestor || isset($_GET['nuevo_gestor'])) : ?>
<form method="post">
    <input type="hidden" name="user_id" value="<?php echo esc_attr($editar_gestor->ID ?? 0); ?>">
    <table class="form-table">
        <tr>
            <th><label for="user_login">Nombre de Usuario:</label></th>
            <td><input type="text" name="user_login" class="regular-text" value="<?php echo esc_attr($editar_gestor->user_login ?? ''); ?>" required></td>
        </tr>
        <tr>
            <th><label for="user_email">Correo electrónico:</label></th>
            <td><input type="email" name="user_email" class="regular-text" value="<?php echo esc_attr($editar_gestor->user_email ?? ''); ?>" required></td>
        </tr>
        <tr>
            <th><label for="user_dni">DNI:</label></th>
            <td><input type="text" name="user_dni" class="regular-text" value="<?php echo esc_attr($editar_gestor->user_dni ?? ''); ?>"></td>
        </tr>

        <tr>
            <th><label for="empresa_id_sel">Empresa (CIF)</label></th>
            <td>
                <select name="empresa_id_sel" id="empresa_id_sel" class="regular-text">
                    <option value="">— Selecciona empresa —</option>
                    <?php
                    // Preselección al editar: leer user_cif_id directamente de tabla users
                    $empresa_sel_id = 0;
                    if (!empty($editar_gestor->ID)) {
                        $empresa_sel_id = (int) $wpdb->get_var($wpdb->prepare(
                            "SELECT user_cif_id FROM $tabla_usuarios WHERE ID = %d",
                            $editar_gestor->ID
                        ));
                    }
                    foreach ($empresas as $e):
                        $label = sprintf('%s (%s) — ID %d', $e->razon_social, $e->user_cif, $e->id);
                    ?>
                        <option value="<?php echo esc_attr($e->id); ?>" <?php selected($empresa_sel_id, (int)$e->id); ?>>
                            <?php echo esc_html($label); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <p class="description">Si seleccionas una empresa, se guardarán su CIF y su ID en el usuario y se creará la vinculación en la tabla de relaciones.</p>
            </td>
        </tr>

        <tr>
            <th><label for="user_cif">CIF (manual)</label></th>
            <td>
                <input type="text" name="user_cif" id="user_cif" class="regular-text" value="<?php echo esc_attr($editar_gestor->user_cif ?? ''); ?>">
                <p class="description">Solo úsalo si <strong>NO</strong> seleccionas empresa.</p>
            </td>
        </tr>

        <tr>
            <th><label for="user_pass">Contraseña:</label></th>
            <td><input type="text" name="user_pass" class="regular-text" placeholder="Dejar en blanco para generar una automática"></td>
        </tr>

        <tr>
            <th><label for="user_gamificacion">Gamificación:</label></th>
            <td>
                <label class="switch">
                    <input type="checkbox" name="user_gamificacion" value="1" <?php echo (!empty($editar_gestor->user_gamificacion)) ? 'checked' : ''; ?>>
                    <span class="slider round"></span>
                </label>
            </td>
        </tr>
    </table>
    <p><input type="submit" name="guardar_gestor" class="button button-primary" value="<?php echo $editar_gestor ? 'Actualizar Gestor' : 'Crear Gestor'; ?>"></p>
</form>
<?php endif; ?>

<h2>Gestores Registrados</h2>
<table class="wp-list-table widefat fixed striped">
    <thead>
        <tr>
            <th>ID</th>
            <th>Nombre de Usuario</th>
            <th>Correo Electrónico</th>
            <th>DNI</th>
            <th>CIF (usuario)</th>
            <th>CIF asociado (por ID)</th>
            <th>Empresa (user_cif_id)</th>
            <th>Gamif. empresa</th>
            <th>Empresas asociadas (IDs)</th>
            <th>Acciones</th>
        </tr>
    </thead>
    <tbody>
        <?php if (empty($gestores)): ?>
            <tr><td colspan="10">No hay gestores.</td></tr>
        <?php else: ?>
            <?php
            // helper de pill
            $pill = function($valor){ 
                $on = strtolower((string)$valor) === 'activado';
                return '<span class="pill '.($on?'on':'off').'">'.($on?'Activado':'Desactivado').'</span>';
            };
            ?>
            <?php foreach ($gestores as $gestor) : ?>
                <tr>
                    <td><?php echo esc_html($gestor->ID); ?></td>
                    <td><?php echo esc_html($gestor->user_login); ?></td>
                    <td><?php echo esc_html($gestor->user_email); ?></td>
                    <td><?php echo esc_html($gestor->user_dni ?: '—'); ?></td>
                    <td><?php echo esc_html($gestor->user_cif ?: '—'); ?></td>
                    <td><?php echo esc_html($gestor->cif_asociado ?: '—'); ?></td>
                    <td style="max-width:260px;">
                        <?php
                        if ($gestor->empresa_asociada) {
                            echo esc_html($gestor->empresa_asociada) . ' ';
                            echo '<span class="small-muted">(ID ' . esc_html((string)$gestor->user_cif_id) . ')</span>';
                        } else {
                            echo '—';
                        }
                        ?>
                    </td>
                    <td><?php echo $gestor->gamif_empresa ? $pill($gestor->gamif_empresa) : '—'; ?></td>
                    <td>
<?php
if (!empty($gestor->empresas_info)) {
    $items = array_filter(array_map('trim', explode('##', $gestor->empresas_info)));
    foreach ($items as $it) {
        list($eid, $cif, $rs) = array_pad(explode('||', $it, 3), 3, '');
        $title = trim(($cif !== '' ? $cif : '—') . ' — ' . ($rs !== '' ? $rs : '—'));
        echo '<span class="gestor-pill" title="' . esc_attr($title) . '">'
           . esc_html($eid)
           . '</span> ';
    }
} else {
    echo '<span class="gestor-pill" title="Sin empresa">—</span>';
}
?>
</td>


                    <td>
                        <a href="?page=control-horario&tab=gestores&editar=<?php echo esc_attr($gestor->ID); ?>" class="button">Editar</a>
                        <a href="?page=control-horario&tab=gestores&eliminar=<?php echo esc_attr($gestor->ID); ?>" class="button button-danger" onclick="return confirm('¿Eliminar este gestor?');">Eliminar</a>
                        <a href="?page=control-horario&tab=gestores&reenviar=<?php echo esc_attr($gestor->ID); ?>" class="button">Reenviar Contraseña</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        <?php endif; ?>
    </tbody>
</table>
<style>
.gestor-pill {
  display:inline-block;
  padding:2px 6px;
  margin:2px 4px 2px 0;
  background:#f1f1f1;
  border-radius:4px;
  font-family:monospace;
  font-size:12px;
  color:#333;
  border:1px solid #ddd;
}
</style>


