<?php
// 🔔 CRON igual que tenías
if (!wp_next_scheduled('enviar_resumen_gestores')) {
    wp_schedule_event(strtotime('next Sunday 20:00'), 'weekly', 'enviar_resumen_gestores');
}
add_action('enviar_resumen_gestores', 'enviar_resumen_semanal_todos_los_gestores');

/** Helper tiempo hh:mm:ss */
if (!function_exists('rh_format_seconds')) {
    function rh_format_seconds($seconds) {
        $seconds = (int)$seconds;
        $h = floor($seconds / 3600);
        $m = floor(($seconds % 3600) / 60);
        $s = $seconds % 60;
        return sprintf('%02d:%02d:%02d', $h, $m, $s);
    }
}

/**
 * 1) Saca todos los gestores ACTIVADOS desde la asociación empresas_usuarios
 */
function enviar_resumen_semanal_todos_los_gestores() {
    global $wpdb;
    $tabla_eu = $wpdb->prefix . 'empresas_usuarios';

    $gestores_ids = $wpdb->get_col("
        SELECT DISTINCT eu.user_id
        FROM {$tabla_eu} eu
        WHERE eu.rol_en_empresa = 'Gestor'
          AND eu.estado = 'Activado'
    ");

    if (!$gestores_ids) return;

    foreach ($gestores_ids as $gestor_id) {
        enviar_resumen_semanal_gestor($gestor_id);
    }
}

/**
 * 2) Para un gestor: por cada empresa que gestiona (eu.rol_en_empresa='Gestor'), manda un email
 *    con el resumen semanal de TODOS los usuarios activados de esa empresa.
 */
function enviar_resumen_semanal_gestor($gestor_id) {
    global $wpdb;

    $tabla_eu = $wpdb->prefix . 'empresas_usuarios';
    $tabla_emp = $wpdb->prefix . 'empresas';
    $tabla_users = $wpdb->prefix . 'users';
    $tabla_rh = $wpdb->prefix . 'registro_horario';
    $tabla_obs = $wpdb->prefix . 'observaciones'; // si no existe, comenta el bloque de observaciones

    // Datos del gestor (email/nombre)
    $gestor = get_userdata($gestor_id);
    if (!$gestor) return;
    $email_destino = $gestor->user_email;
    $nombre_gestor = $gestor->display_name;

    // Rango semanal (zona WP)
    $tz = wp_timezone();
    $inicio_dt = new DateTime('monday this week 00:00:00', $tz);
    $fin_dt    = new DateTime('monday next week 00:00:00', $tz);
    $inicio_semana = $inicio_dt->format('Y-m-d H:i:s');
    $fin_semana    = $fin_dt->format('Y-m-d H:i:s');

    // Empresas que gestiona este usuario
    $empresas_ids = $wpdb->get_col($wpdb->prepare("
        SELECT eu.empresa_id
        FROM {$tabla_eu} eu
        WHERE eu.user_id = %d
          AND eu.rol_en_empresa = 'Gestor'
          AND eu.estado = 'Activado'
    ", $gestor_id));

    if (!$empresas_ids) return;

    // Días de la semana (clave en inglés para date('l'))
    $dias_semana = [
        'Monday'    => 'Lunes',
        'Tuesday'   => 'Martes',
        'Wednesday' => 'Miércoles',
        'Thursday'  => 'Jueves',
        'Friday'    => 'Viernes',
        'Saturday'  => 'Sábado',
        'Sunday'    => 'Domingo'
    ];

    foreach ($empresas_ids as $empresa_id) {

       // Info empresa: CIF, nombre y ESTADO
$empresa = $wpdb->get_row($wpdb->prepare("
    SELECT id, user_cif, razon_social, estado_empresa
    FROM {$tabla_emp}
    WHERE id = %d
", $empresa_id));

if (!$empresa || $empresa->estado_empresa !== 'activado') {
    // 👋 Empresa desactivada: NO enviar resumen
    continue;
}


        // Trabajadores de la empresa (activados). Incluye Gestor/Técnico/Trabajador si quieres verlos en la tabla.
        $trabajadores = $wpdb->get_results($wpdb->prepare("
            SELECT u.ID, u.display_name
            FROM {$tabla_eu} eu
            JOIN {$tabla_users} u ON u.ID = eu.user_id
            WHERE eu.empresa_id = %d
              AND eu.estado = 'Activado'
              AND eu.rol_en_empresa IN ('Trabajador','Gestor','Técnico')
              AND (u.user_estado IS NULL OR u.user_estado = 'Activado')
            ORDER BY u.display_name ASC
        ", $empresa_id));

        $resumenHTML = '';
        $observacionesHTML = '';

        if ($trabajadores) {
            foreach ($trabajadores as $t) {
                $user_id = (int)$t->ID;
                $nombre  = esc_html($t->display_name);

                // Sumar segundos por día para ese user y esa empresa (cif_id = empresa_id)
                $registros = $wpdb->get_results($wpdb->prepare("
    SELECT DATE(COALESCE(entradaR, entrada)) AS dia,
           SUM(TIME_TO_SEC(COALESCE(tiempo_totalR, tiempo_total))) AS total_seg_dia
    FROM {$tabla_rh}
    WHERE user_id = %d
      AND (
            cif_id = %d
         OR (cif_id IS NULL AND cif_usuario = %s)  -- fallback si no rellenaste cif_id
      )
      AND COALESCE(entradaR, entrada) >= %s
      AND COALESCE(entradaR, entrada) < %s
      AND COALESCE(tiempo_totalR, tiempo_total) IS NOT NULL
    GROUP BY DATE(COALESCE(entradaR, entrada))
", $user_id, $empresa_id, $empresa->user_cif, $inicio_semana, $fin_semana));


                $tiempo_por_dia = [];
                $tiempo_total_seg = 0;

                foreach ($registros as $r) {
                    $seg = (int)$r->total_seg_dia;
                    $tiempo_total_seg += $seg;
$dia_php = date('l', strtotime($r->dia));
                    $tiempo_por_dia[$dia_php] = $seg;
                }

                $resumenHTML .= "<tr style='border-bottom:1px solid #ccc;'>
                    <td style='padding:10px;text-align:center;'>{$nombre}</td>";

                foreach (array_keys($dias_semana) as $diaEN) {
                    $dur = isset($tiempo_por_dia[$diaEN]) ? rh_format_seconds($tiempo_por_dia[$diaEN]) : '&mdash;';
                    $resumenHTML .= "<td style='padding:10px;text-align:center;'>{$dur}</td>";
                }

                $total_txt = $tiempo_total_seg > 0 ? rh_format_seconds($tiempo_total_seg) : '&mdash;';
                $resumenHTML .= "<td style='padding:10px;text-align:center;font-weight:bold;'>{$total_txt}</td></tr>";

                // Observaciones (si existe tabla)
                if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $tabla_obs))) {
                    $obs = $wpdb->get_results($wpdb->prepare("
                        SELECT fecha_observacion, observaciones
                        FROM {$tabla_obs}
                        WHERE user_id = %d
                          AND fecha_observacion BETWEEN %s AND %s
                        ORDER BY fecha_observacion ASC
                    ", $user_id, $inicio_dt->format('Y-m-d'), $fin_dt->format('Y-m-d')));

                    if ($obs) {
                        $observacionesHTML .= "<h4 style='margin-top:30px;'>Observaciones de {$nombre}</h4><ul style='padding-left:20px;'>";
                        foreach ($obs as $o) {
                            $fecha = date('d/m/Y', strtotime($o->fecha_observacion));
                            $observacionesHTML .= "<li><strong>{$fecha}:</strong> " . esc_html($o->observaciones) . "</li>";
                        }
                        $observacionesHTML .= '</ul>';
                    }
                }
            }
        }

        // Montar tabla
        $thead = "<tr><th style='padding:10px;text-align:center;'>Nombre</th>";
        foreach ($dias_semana as $dES) {
            $thead .= "<th style='padding:10px;text-align:center;'>{$dES}</th>";
        }
        $thead .= "<th style='padding:10px;text-align:center;'>Total Semana</th></tr>";

        $tablaResumen = "<table style='width:100%; border-collapse: collapse; margin-top: 20px;'>
            <thead style='background-color:#0073aa; color:#fff;'>{$thead}</thead>
            <tbody>{$resumenHTML}</tbody>
        </table>";

        // Cargar plantilla
        $template_path = WP_PLUGIN_DIR . '/control-horario-trabajadores/templates/resumen_semanal_empresa.html';
        if (!file_exists($template_path)) continue; // sin plantilla, salta esta empresa

        $html = file_get_contents($template_path);

        $rango = $inicio_dt->format('d/m/Y') . ' al ' . $inicio_dt->modify('+6 days')->format('d/m/Y'); // L a D
        // Restauramos $inicio_dt si lo quieres después (no hace falta aquí)

        // Reemplazos
        $html = str_replace(
            ['${nombreGestor}', '${cifEmpresa}', '${rangoSemana}', '${tablaResumenHTML}', '${tablaObservacionesHTML}', '${nombreEmpresa}'],
            [$nombre_gestor, $empresa->user_cif, $rango, $tablaResumen, $observacionesHTML, ($empresa->razon_social ?: '')],
            $html
        );

        // Envío
        $headers = ['Content-Type: text/html; charset=UTF-8'];
        $asunto = "📊 Resumen semanal de actividad - {$empresa->user_cif}" . ($empresa->razon_social ? " ({$empresa->razon_social})" : '');
        wp_mail($email_destino, $asunto, $html, $headers);
    }
}

/**
 * Gatillo manual para pruebas (admin):
 * /?resumen_gestor=123
 */
add_action('init', function () {
    if (isset($_GET['resumen_gestor']) && current_user_can('manage_options')) {
        $gestor_id = (int) $_GET['resumen_gestor'];
        enviar_resumen_semanal_gestor($gestor_id);
        echo "✅ Resumen semanal enviado al gestor con ID $gestor_id";
        exit;
    }
});
