<?php
if (!defined('ABSPATH')) exit;

/**
 * Procesar aprobación / rechazo de solicitudes de cambio de fichaje por parte del Gestor
 */
function rh_procesar_validacion_cambio_fichaje_gestor() {
    if (!is_user_logged_in()) {
        return;
    }

    if (empty($_POST['accion_cambio_gestor'])) {
        return;
    }

    global $wpdb;

    $user_id = get_current_user_id();

    // Comprobar que es Gestor + Activado
    $gestor = $wpdb->get_row($wpdb->prepare(
        "SELECT user_rol, user_estado FROM {$wpdb->prefix}users WHERE ID = %d",
        $user_id
    ));

    if (!$gestor || $gestor->user_rol !== 'Gestor' || $gestor->user_estado !== 'Activado') {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_sin_permiso', remove_query_arg('rh_msg')));
        exit;
    }

    // Seguridad nonce
    if (
        !isset($_POST['cambio_gestor_nonce']) ||
        !wp_verify_nonce($_POST['cambio_gestor_nonce'], 'cambio_gestor_nonce_action')
    ) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_error_seguridad', remove_query_arg('rh_msg')));
        exit;
    }

    $accion       = sanitize_text_field($_POST['accion_cambio_gestor']); // aprobar | rechazar
    $solicitud_id = isset($_POST['solicitud_id']) ? intval($_POST['solicitud_id']) : 0;

    if (!$solicitud_id || !in_array($accion, ['aprobar','rechazar'], true)) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_datos_invalidos', remove_query_arg('rh_msg')));
        exit;
    }

    $tabla_solicitudes = $wpdb->prefix . 'solicitudes';
    $tabla_registro    = $wpdb->prefix . 'registro_horario';
    $tabla_observ      = $wpdb->prefix . 'observaciones';

    // Obtener todos los CIFs donde el usuario es Gestor
    $cifs_gestor = $wpdb->get_col($wpdb->prepare(
        "SELECT DISTINCT e.user_cif
         FROM {$wpdb->prefix}empresas_usuarios eu
         JOIN {$wpdb->prefix}empresas e ON e.id = eu.empresa_id
         WHERE eu.user_id = %d
           AND eu.rol_en_empresa = 'Gestor'
           AND eu.estado = 'Activado'",
        $user_id
    ));

    if (empty($cifs_gestor)) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_sin_empresas', remove_query_arg('rh_msg')));
        exit;
    }

    // Cargar solicitud
    $solicitud = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $tabla_solicitudes WHERE id = %d",
        $solicitud_id
    ));

    if (
        !$solicitud ||
        $solicitud->estado !== 'pendiente' ||
        strpos($solicitud->motivo, 'CAMBIO_FICHAJE') !== 0
    ) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_solicitud_no_valida', remove_query_arg('rh_msg')));
        exit;
    }

    // Comprobar que la solicitud pertenece a un CIF gestionado por este Gestor
    if (!in_array($solicitud->cif_usuario, $cifs_gestor, true)) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_otro_cif', remove_query_arg('rh_msg')));
        exit;
    }

    // Obtener ID de registro desde el motivo
    $registro_id = 0;
    if (preg_match('/REGISTRO_ID=(\d+)/', $solicitud->motivo, $m)) {
        $registro_id = intval($m[1]);
    }
    if (!$registro_id) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_parse_error', remove_query_arg('rh_msg')));
        exit;
    }

    $gestor_user   = wp_get_current_user();
    $gestor_nombre = $gestor_user ? $gestor_user->display_name : ('Gestor ' . $user_id);
    $cif_solicitud = $solicitud->cif_usuario; // CIF literal
    $trabajador_id = (int) $solicitud->user_id;
    $fecha_fichaje = $solicitud->fecha_inicio ?: current_time('Y-m-d');
    $ts_now        = current_time('d/m/Y H:i:s');

    // Helper para cortar observaciones a 250 chars
    $trim_obs = function($text) {
        if (function_exists('mb_substr')) {
            return mb_substr($text, 0, 250, 'UTF-8');
        }
        return substr($text, 0, 250);
    };

    // Si solo RECHAZA, no tocamos el registro de fichaje
    if ($accion === 'rechazar') {
        $texto_obs = sprintf(
            '%s [Cambio fichaje RECHAZADO por %s] Reg %d',
            $ts_now,
            $gestor_nombre,
            $registro_id
        );

        $wpdb->insert($tabla_observ, [
            'user_id'          => $trabajador_id,
            'cif_usuario'      => $cif_solicitud,
            'fecha_observacion'=> $fecha_fichaje,
            'observaciones'    => $trim_obs($texto_obs),
        ]);

        $wpdb->update($tabla_solicitudes, [
            'estado'        => 'rechazado',
            'gestionado_por'=> $user_id,
            'fecha_gestion' => current_time('mysql'),
        ], [
            'id' => $solicitud_id
        ]);

        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_rechazado_ok', remove_query_arg('rh_msg')));
        exit;
    }

    // --- A partir de aquí: APROBAR CAMBIO ---

    $motivo = $solicitud->motivo;

    // Formato del motivo que generamos:
    // "CAMBIO_FICHAJE | REGISTRO_ID=X | Trabajador ... del fichaje del día dd/mm/YYYY:
    //  de HH:MM - HH:MM a HH:MM - HH:MM. Motivo: ..."
    $patron = '/del fichaje del día ([0-9]{2}\/[0-9]{2}\/[0-9]{4}): de ([0-9]{2}:[0-9]{2}|—) - ([0-9]{2}:[0-9]{2}|—) a ([0-9]{2}:[0-9]{2}|—) - ([0-9]{2}:[0-9]{2}|—)\. Motivo: (.*)$/u';

    if (!preg_match($patron, $motivo, $m2)) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_parse_error', remove_query_arg('rh_msg')));
        exit;
    }

    // $m2[1] = fecha_txt (no la usamos, usamos fecha_inicio de la solicitud)
    // $m2[4] = entrada_nueva, $m2[5] = salida_nueva
    $entrada_nueva_hm = $m2[4]; // HH:MM
    $salida_nueva_hm  = $m2[5]; // HH:MM o '—'

    if ($entrada_nueva_hm === '—') {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_entrada_vacia', remove_query_arg('rh_msg')));
        exit;
    }

    $entrada_nueva_dt = $fecha_fichaje . ' ' . $entrada_nueva_hm . ':00';
    $salida_nueva_dt  = ($salida_nueva_hm !== '—') ? ($fecha_fichaje . ' ' . $salida_nueva_hm . ':00') : null;

    $ts_entrada = strtotime($entrada_nueva_dt);
    $ts_salida  = $salida_nueva_dt ? strtotime($salida_nueva_dt) : false;

    if (!$ts_entrada || ($ts_salida && $ts_salida <= $ts_entrada)) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_horas_invalidas', remove_query_arg('rh_msg')));
        exit;
    }

    // Cargar registro de fichaje
    $registro = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $tabla_registro WHERE id = %d",
        $registro_id
    ));

    if (!$registro) {
        wp_safe_redirect(add_query_arg('rh_msg', 'gestor_registro_no_encontrado', remove_query_arg('rh_msg')));
        exit;
    }

    // Calcular tiempo total
    $tiempo_total_txt = '00:00:00';
    if ($ts_salida) {
        $diff = max(0, $ts_salida - $ts_entrada);
        $h = floor($diff / 3600);
        $m = floor(($diff % 3600) / 60);
        $s = $diff % 60;
        $tiempo_total_txt = sprintf('%02d:%02d:%02d', $h, $m, $s);
    }

    // Actualizar registro de horario (columnas base + R)
    $datos_update = [
        'entradaR'      => $entrada_nueva_dt,
        'entrada'       => $entrada_nueva_dt,
        'tiempo_totalR' => $tiempo_total_txt,
        'tiempo_total'  => $tiempo_total_txt,
    ];

    if ($ts_salida) {
        $datos_update['salidaR'] = $salida_nueva_dt;
        $datos_update['salida']  = $salida_nueva_dt;
    } else {
        $datos_update['salidaR'] = null;
        $datos_update['salida']  = null;
    }

    $wpdb->update($tabla_registro, $datos_update, [
        'id' => $registro_id
    ]);

    // Insertar observación de APROBACIÓN
    $texto_obs = sprintf(
        '%s [Cambio fichaje APROBADO por %s] Reg %d (%s-%s)',
        $ts_now,
        $gestor_nombre,
        $registro_id,
        $entrada_nueva_hm,
        ($salida_nueva_hm !== '—' ? $salida_nueva_hm : 'sin salida')
    );

    $wpdb->insert($tabla_observ, [
        'user_id'          => $trabajador_id,
        'cif_usuario'      => $cif_solicitud,   // CIF literal
        'fecha_observacion'=> $fecha_fichaje,
        'observaciones'    => $trim_obs($texto_obs),
    ]);

    // Marcar solicitud como aprobada
    $wpdb->update($tabla_solicitudes, [
        'estado'        => 'aprobado',
        'gestionado_por'=> $user_id,
        'fecha_gestion' => current_time('mysql'),
    ], [
        'id' => $solicitud_id
    ]);

    wp_safe_redirect(add_query_arg('rh_msg', 'gestor_aprobado_ok', remove_query_arg('rh_msg')));
    exit;
}
add_action('init', 'rh_procesar_validacion_cambio_fichaje_gestor');


/**
 * Shortcode: [revision_cambio_horario]
 * Revisión de solicitudes de cambio de fichaje (solo Gestor)
 */
/**
 * Shortcode: [revision_cambio_horario]
 * Revisión de solicitudes de cambio de fichaje (solo Gestor)
 */
function rh_shortcode_revision_cambio_horario() {
    if (!is_user_logged_in()) {
        return '<div class="container-plugin"><p class="pure-alert">Debes iniciar sesión.</p></div>';
    }

    global $wpdb;
    $user_id = get_current_user_id();

    // Verificar rol Gestor + Activado
    $gestor = $wpdb->get_row($wpdb->prepare(
        "SELECT user_rol, user_estado FROM {$wpdb->prefix}users WHERE ID = %d",
        $user_id
    ));
    if (!$gestor || $gestor->user_rol !== 'Gestor' || $gestor->user_estado !== 'Activado') {
        return '<div class="container-plugin"><p class="pure-alert">No tienes permisos para acceder a esta sección.</p></div>';
    }

    // Empresas del gestor (TODAS, no se selecciona una)
    $empresas = $wpdb->get_results($wpdb->prepare(
        "SELECT DISTINCT e.id, e.user_cif, e.razon_social
         FROM {$wpdb->prefix}empresas_usuarios eu
         JOIN {$wpdb->prefix}empresas e ON e.id = eu.empresa_id
         WHERE eu.user_id = %d
           AND eu.rol_en_empresa = 'Gestor'
           AND eu.estado = 'Activado'
         ORDER BY e.razon_social ASC, e.user_cif ASC",
        $user_id
    ));

    if (!$empresas) {
        return '<div class="container-plugin"><p class="pure-alert" style="text-align:center;">No tienes ninguna empresa asociada como Gestor.</p></div>';
    }

    // Array de CIFs gestionados
    $cifs_gestor = [];
    foreach ($empresas as $e) {
        $cifs_gestor[] = $e->user_cif;
    }

    // --------- Filtros ---------
    $estado_filtro = isset($_GET['estado']) ? sanitize_text_field($_GET['estado']) : 'pendiente';
    $desde         = isset($_GET['desde'])  ? sanitize_text_field($_GET['desde'])  : '';
    $hasta         = isset($_GET['hasta'])  ? sanitize_text_field($_GET['hasta'])  : '';
    $trabajador_id = isset($_GET['trabajador_id']) ? intval($_GET['trabajador_id']) : 0;

    $tabla_solicitudes = $wpdb->prefix . 'solicitudes';

    // WHERE con IN para todos los CIF del gestor
    $placeholders = implode(',', array_fill(0, count($cifs_gestor), '%s'));
    $whereParts   = [];
    $params       = [];

    // cif_usuario IN (...)
    $whereParts[] = "s.cif_usuario IN ($placeholders)";
    $params       = array_merge($params, $cifs_gestor);

    // Solo solicitudes de cambio de fichaje
    $whereParts[] = "s.motivo LIKE 'CAMBIO_FICHAJE%'";
    
    if ($estado_filtro !== 'todos') {
        $whereParts[] = "s.estado = %s";
        $params[]     = $estado_filtro;
    }
    if ($desde) {
        $whereParts[] = "s.fecha_inicio >= %s";
        $params[]     = $desde;
    }
    if ($hasta) {
        $whereParts[] = "s.fecha_fin <= %s";
        $params[]     = $hasta;
    }
    if ($trabajador_id) {
        $whereParts[] = "s.user_id = %d";
        $params[]     = $trabajador_id;
    }

    $whereSql = implode(' AND ', $whereParts);

    // JOIN con empresas para mostrar el nombre en la tabla
    $sql = "
        SELECT s.*, e.razon_social, e.user_cif
        FROM $tabla_solicitudes s
        LEFT JOIN {$wpdb->prefix}empresas e
               ON e.user_cif = s.cif_usuario
        WHERE $whereSql
        ORDER BY s.fecha_solicitud DESC
    ";
    $solicitudes = $wpdb->get_results($wpdb->prepare($sql, $params));

    // Lista de trabajadores de TODAS las empresas del gestor (para el filtro)
    $in_cifs = "'" . implode("','", array_map('esc_sql', $cifs_gestor)) . "'";
    $trabajadores = $wpdb->get_results("
        SELECT ID, display_name
        FROM {$wpdb->prefix}users
        WHERE user_cif IN ($in_cifs)
          AND user_estado = 'Activado'
          AND user_rol IN ('Gestor','Trabajador')
        ORDER BY display_name ASC
    ");

    // --------- Mensajes ---------
    $mensaje_html = '';
    if (!empty($_GET['rh_msg'])) {
        switch ($_GET['rh_msg']) {
            case 'gestor_aprobado_ok':
                $mensaje_html = '<div class="alert alert-success" style="text-align:center;">
                    ✅ Cambio aplicado correctamente.
                </div>';
                break;
            case 'gestor_rechazado_ok':
                $mensaje_html = '<div class="alert" style="text-align:center; background:#e2e3ff; color:#383d7c; border:1px solid #c5c6f7;">
                    ℹ️ Solicitud rechazada.
                </div>';
                break;
            case 'gestor_sin_permiso':
            case 'gestor_error_seguridad':
            case 'gestor_datos_invalidos':
            case 'gestor_solicitud_no_valida':
            case 'gestor_otro_cif':
            case 'gestor_parse_error':
            case 'gestor_entrada_vacia':
            case 'gestor_horas_invalidas':
            case 'gestor_registro_no_encontrado':
            case 'gestor_sin_empresas':
                $mensaje_html = '<div class="alert alert-danger" style="text-align:center;">
                    ⚠️ No se ha podido procesar la solicitud. Revisa los datos.
                </div>';
                break;
        }
    }

    ob_start();
    ?>
    <div class="container-plugin">
        <?php echo $mensaje_html; ?>

        <h3>Revisión de solicitudes de cambio de fichaje</h3>

        <p style="font-size: 12px; color:#666; margin-top:-10px; margin-bottom:15px;">
            Mostrando solicitudes de todas las empresas que gestionas.
        </p>

        <!-- Filtros -->
        <form method="get" class="pure-form pure-form-stacked" style="margin-bottom:15px;">
            <fieldset style="display:flex; flex-wrap:wrap; gap:10px; align-items:flex-end;">
                <div style="min-width:180px;">
                    <label for="estado">Estado</label>
                    <select name="estado" id="estado">
                        <option value="pendiente" <?php selected($estado_filtro, 'pendiente'); ?>>Pendientes</option>
                        <option value="aprobado"  <?php selected($estado_filtro, 'aprobado'); ?>>Aprobadas</option>
                        <option value="rechazado"<?php selected($estado_filtro, 'rechazado'); ?>>Rechazadas</option>
                        <option value="todos"     <?php selected($estado_filtro, 'todos'); ?>>Todas</option>
                    </select>
                </div>

                <div style="min-width:150px;">
                    <label for="desde">Desde</label>
                    <input type="date" name="desde" id="desde" value="<?php echo esc_attr($desde); ?>">
                </div>

                <div style="min-width:150px;">
                    <label for="hasta">Hasta</label>
                    <input type="date" name="hasta" id="hasta" value="<?php echo esc_attr($hasta); ?>">
                </div>

                <div style="min-width:220px;">
                    <label for="trabajador_id">Trabajador</label>
                    <select name="trabajador_id" id="trabajador_id">
                        <option value="0">Todos</option>
                        <?php foreach ($trabajadores as $t): ?>
                            <option value="<?php echo esc_attr($t->ID); ?>" <?php selected($trabajador_id, $t->ID); ?>>
                                <?php echo esc_html($t->display_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div>
                    <button type="submit" class="pure-button pure-button-primary">Filtrar</button>
                </div>
            </fieldset>
        </form>

        <?php if ($solicitudes): ?>
            <div class="tabla-responsive">
                <table class="pure-table tabla-solicitudes">
                    <thead>
                        <tr>
                            <th>Empresa</th>
                            <th>Trabajador</th>
                            <th>Fecha fichaje</th>
                            <th>Solicitud</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($solicitudes as $s): 
                        $trabajador   = get_userdata($s->user_id);
                        $nombre_trab  = $trabajador ? $trabajador->display_name : '—';
                        $fecha_fichaje= $s->fecha_inicio ?: '';
                        $fecha_txt    = $fecha_fichaje ? date('d/m/Y', strtotime($fecha_fichaje)) : '—';

                        $empresa_label = $s->razon_social 
                            ? ($s->razon_social . ' (' . $s->user_cif . ')')
                            : $s->user_cif;

                        $estado = strtolower($s->estado);
                        $color  = '#555';
                        if ($estado === 'aprobado')  $color = 'green';
                        if ($estado === 'rechazado') $color = 'red';

                        // Preview corto del motivo
                        $motivo_full = $s->motivo;
                        $motivo_sin_prefix = preg_replace('/^CAMBIO_FICHAJE \| REGISTRO_ID=\d+ \| /', '', $motivo_full);
                        if (function_exists('mb_substr')) {
                            $motivo_preview = mb_substr($motivo_sin_prefix, 0, 180, 'UTF-8');
                        } else {
                            $motivo_preview = substr($motivo_sin_prefix, 0, 180);
                        }
                        if (strlen($motivo_sin_prefix) > strlen($motivo_preview)) {
                            $motivo_preview .= '…';
                        }
                    ?>
                        <tr>
                            <td><?php echo esc_html($empresa_label); ?></td>
                            <td><?php echo esc_html($nombre_trab); ?></td>
                            <td><?php echo esc_html($fecha_txt); ?></td>
                            <td style="text-align:left;" title="<?php echo esc_attr($motivo_full); ?>">
                                <?php echo esc_html($motivo_preview); ?>
                            </td>
                            <td>
                                <span style="color:<?php echo esc_attr($color); ?>; font-weight:bold;">
                                    <?php echo ucfirst($estado); ?>
                                </span>
                            </td>
                            <td class="acciones">
                                <?php if ($estado === 'pendiente'): ?>
                                    <div class="acciones">
                                        <form method="post" style="display:inline;">
                                            <?php wp_nonce_field('cambio_gestor_nonce_action', 'cambio_gestor_nonce'); ?>
                                            <input type="hidden" name="solicitud_id" value="<?php echo esc_attr($s->id); ?>">
                                            <button type="submit"
                                                    name="accion_cambio_gestor"
                                                    value="aprobar"
                                                    class="pure-button pure-button-success">
                                                Aprobar
                                            </button>
                                        </form>

                                        <form method="post" style="display:inline;">
                                            <?php wp_nonce_field('cambio_gestor_nonce_action', 'cambio_gestor_nonce'); ?>
                                            <input type="hidden" name="solicitud_id" value="<?php echo esc_attr($s->id); ?>">
                                            <button type="submit"
                                                    name="accion_cambio_gestor"
                                                    value="rechazar"
                                                    class="pure-button pure-button-danger">
                                                Rechazar
                                            </button>
                                        </form>
                                    </div>
                                <?php else: ?>
                                    <span style="font-size:12px; color:#777;">—</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php else: ?>
            <p>No hay solicitudes que coincidan con los filtros seleccionados.</p>
        <?php endif; ?>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode('revision_cambio_horario', 'rh_shortcode_revision_cambio_horario');
