<?php
if (!defined('ABSPATH')) exit;

/**
 * Procesar solicitudes de cambio de fichaje
 */
function rh_procesar_solicitud_cambio_fichaje() {
    if (!is_user_logged_in()) {
        return;
    }

    if (empty($_POST['accion']) || $_POST['accion'] !== 'solicitar_cambio_fichaje') {
        return;
    }

    // Seguridad nonce
    if (
        !isset($_POST['cambio_fichaje_nonce']) ||
        !wp_verify_nonce($_POST['cambio_fichaje_nonce'], 'cambio_fichaje_nonce_action')
    ) {
        wp_safe_redirect(add_query_arg('rh_msg', 'cambio_error_seguridad', remove_query_arg('rh_msg')));
        exit;
    }

    global $wpdb;

    $user_id   = get_current_user_id();
    $user_info = get_userdata($user_id);

    $tabla_registro    = $wpdb->prefix . 'registro_horario';
    $tabla_solicitudes = $wpdb->prefix . 'solicitudes';
    $tabla_users       = $wpdb->prefix . 'users';

    $registro_id    = isset($_POST['registro_id']) ? intval($_POST['registro_id']) : 0;
    $entrada_nueva  = isset($_POST['entrada_nueva']) ? sanitize_text_field($_POST['entrada_nueva']) : '';
    $salida_nueva   = isset($_POST['salida_nueva']) ? sanitize_text_field($_POST['salida_nueva']) : '';
    $motivo_usuario = isset($_POST['motivo']) ? sanitize_textarea_field($_POST['motivo']) : '';

    if (!$registro_id || !$entrada_nueva) {
        wp_safe_redirect(add_query_arg('rh_msg', 'cambio_datos_invalidos', remove_query_arg('rh_msg')));
        exit;
    }

    // Formato esperado: "YYYY-MM-DDTHH:MM"
    $ts_entrada_nueva = strtotime($entrada_nueva);
    $ts_salida_nueva  = $salida_nueva ? strtotime($salida_nueva) : false;

    if (!$ts_entrada_nueva) {
        wp_safe_redirect(add_query_arg('rh_msg', 'cambio_datos_invalidos', remove_query_arg('rh_msg')));
        exit;
    }

    if ($ts_salida_nueva && $ts_salida_nueva <= $ts_entrada_nueva) {
        wp_safe_redirect(add_query_arg('rh_msg', 'cambio_salida_anterior', remove_query_arg('rh_msg')));
        exit;
    }

    // Ver que el registro pertenece al usuario
    $registro = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM $tabla_registro WHERE id = %d AND user_id = %d",
        $registro_id,
        $user_id
    ));

    if (!$registro) {
        wp_safe_redirect(add_query_arg('rh_msg', 'cambio_registro_no_encontrado', remove_query_arg('rh_msg')));
        exit;
    }

    // CIF literal del usuario (igual que en mnkt_observaciones.cif_usuario)
    $cif_usuario = $wpdb->get_var($wpdb->prepare(
        "SELECT user_cif FROM $tabla_users WHERE ID = %d",
        $user_id
    ));

    // DNI usuario
    $user_dni = $wpdb->get_var($wpdb->prepare(
        "SELECT user_dni FROM $tabla_users WHERE ID = %d",
        $user_id
    ));

    $nombre_trab = $user_info ? $user_info->display_name : '';

    // Datos originales
    $entrada_original_dt = $registro->entradaR ?: $registro->entrada;
    $salida_original_dt  = $registro->salidaR  ?: $registro->salida;

    $fecha_fichaje_date = $entrada_original_dt ? date('Y-m-d', strtotime($entrada_original_dt)) : date('Y-m-d');
    $fecha_fichaje_txt  = date('d/m/Y', strtotime($fecha_fichaje_date));

    $entrada_original_str = $entrada_original_dt ? date('H:i', strtotime($entrada_original_dt)) : '—';
    $salida_original_str  = $salida_original_dt  ? date('H:i', strtotime($salida_original_dt))  : '—';

    $entrada_nueva_str = date('H:i', $ts_entrada_nueva);
    $salida_nueva_str  = $ts_salida_nueva ? date('H:i', $ts_salida_nueva) : '—';

    // Evitar duplicar solicitudes para el mismo registro:
    // Buscamos en solicitudes pendientes que tengan REGISTRO_ID=X en el motivo
    $solicitudes_pendientes = $wpdb->get_results($wpdb->prepare(
        "SELECT motivo 
         FROM $tabla_solicitudes 
         WHERE user_id = %d 
           AND cif_usuario = %s
           AND estado = 'pendiente'
           AND motivo LIKE 'CAMBIO_FICHAJE%%'",
        $user_id,
        $cif_usuario
    ));

    if ($solicitudes_pendientes) {
        foreach ($solicitudes_pendientes as $sol) {
            if (preg_match('/REGISTRO_ID=(\d+)/', $sol->motivo, $m)) {
                if (intval($m[1]) === $registro_id) {
                    wp_safe_redirect(add_query_arg('rh_msg', 'cambio_ya_pendiente', remove_query_arg('rh_msg')));
                    exit;
                }
            }
        }
    }

    // Texto completo en motivo (todo va aquí)
    $motivo_final = sprintf(
        "CAMBIO_FICHAJE | REGISTRO_ID=%d | Trabajador %s (DNI %s) solicita el cambio del fichaje del día %s: de %s - %s a %s - %s. Motivo: %s",
        $registro_id,
        $nombre_trab ?: 'Sin nombre',
        $user_dni ?: 'Sin DNI',
        $fecha_fichaje_txt,
        $entrada_original_str,
        $salida_original_str,
        $entrada_nueva_str,
        $salida_nueva_str,
        $motivo_usuario ?: 'Sin motivo'
    );

    // Tipo: si no quieres tocar el enum, usa uno de los existentes (p.ej. 'permiso')
    $tipo_solicitud = 'permiso';

    $wpdb->insert($tabla_solicitudes, [
        'user_id'         => $user_id,
        'cif_usuario'     => $cif_usuario,          // ← CIF literal, mismo criterio que mnkt_observaciones
        'tipo'            => $tipo_solicitud,
        'fecha_inicio'    => $fecha_fichaje_date,
        'fecha_fin'       => $fecha_fichaje_date,
        'motivo'          => $motivo_final,
        'estado'          => 'pendiente',
        'fecha_solicitud' => current_time('mysql'),
    ]);

    wp_safe_redirect(add_query_arg('rh_msg', 'cambio_ok', remove_query_arg('rh_msg')));
    exit;
}
add_action('init', 'rh_procesar_solicitud_cambio_fichaje');


/**
 * Shortcode: [registro_cambio_horario]
 * Listado de fichajes del usuario + solicitud de cambio
 */
function rh_shortcode_registro_cambio_horario() {
    if (!is_user_logged_in()) {
        return '<p class="pure-alert" style="text-align:center;">Debes iniciar sesión para ver tus fichajes.</p>';
    }

    global $wpdb;

    $user_id   = get_current_user_id();
    $user_info = get_userdata($user_id);

    $tabla_registro    = $wpdb->prefix . 'registro_horario';
    $tabla_solicitudes = $wpdb->prefix . 'solicitudes';
    $tabla_users       = $wpdb->prefix . 'users';

    // No dejar entrar a inspectores si usas ese rol
    if (in_array('inspector', (array) $user_info->roles, true)) {
        return '<p class="pure-alert" style="text-align:center;">No tienes acceso a esta sección.</p>';
    }

    // CIF literal del usuario (igual que en mnkt_observaciones.cif_usuario)
    $cif_usuario = $wpdb->get_var($wpdb->prepare(
        "SELECT user_cif FROM $tabla_users WHERE ID = %d",
        $user_id
    ));

    // Mensajes
    $mensaje_html = '';
    if (!empty($_GET['rh_msg'])) {
        switch ($_GET['rh_msg']) {
            case 'cambio_ok':
                $mensaje_html = '<div style="margin:10px 0;padding:10px;border-radius:5px;background:#d4edda;color:#155724;text-align:center;">
                    ✅ Solicitud de cambio enviada. Pendiente de revisión.
                </div>';
                break;
            case 'cambio_ya_pendiente':
                $mensaje_html = '<div style="margin:10px 0;padding:10px;border-radius:5px;background:#fff3cd;color:#856404;text-align:center;">
                    ⚠️ Ya existe una solicitud pendiente para ese fichaje.
                </div>';
                break;
            case 'cambio_registro_no_encontrado':
            case 'cambio_datos_invalidos':
            case 'cambio_salida_anterior':
            case 'cambio_error_seguridad':
                $mensaje_html = '<div style="margin:10px 0;padding:10px;border-radius:5px;background:#f8d7da;color:#721c24;text-align:center;">
                    ⚠️ No se ha podido crear la solicitud de cambio. Revisa los datos.
                </div>';
                break;
        }
    }

    // Buscar solicitudes pendientes tipo CAMBIO_FICHAJE y sacar REGISTRO_ID
    $registros_pendientes = [];
    if ($cif_usuario) {
        $sol_pend = $wpdb->get_results($wpdb->prepare(
            "SELECT motivo 
             FROM $tabla_solicitudes
             WHERE user_id = %d
               AND cif_usuario = %s
               AND estado = 'pendiente'
               AND motivo LIKE 'CAMBIO_FICHAJE%%'",
            $user_id,
            $cif_usuario
        ));
        if ($sol_pend) {
            foreach ($sol_pend as $sol) {
                if (preg_match('/REGISTRO_ID=(\d+)/', $sol->motivo, $m)) {
                    $registros_pendientes[] = intval($m[1]);
                }
            }
        }
    }

    // Filtros de fechas
    $hoy          = current_time('Y-m-d');
    $ayer         = date('Y-m-d', strtotime($hoy . ' -1 day'));
    $lunes_semana = date('Y-m-d', strtotime('monday this week', strtotime($hoy)));
    $domingo_sem  = date('Y-m-d', strtotime('sunday this week', strtotime($hoy)));
    $inicio_mes   = date('Y-m-01', strtotime($hoy));

    $filtro = isset($_GET['filtro']) ? sanitize_text_field($_GET['filtro']) : 'hoy';

    $condicion_registros = '';
    switch ($filtro) {
        case 'hoy':
            $condicion_registros = $wpdb->prepare("AND DATE(entradaR) = %s", $hoy);
            break;
        case 'ayer':
            $condicion_registros = $wpdb->prepare("AND DATE(entradaR) = %s", $ayer);
            break;
        case 'semana':
            $condicion_registros = $wpdb->prepare("AND DATE(entradaR) BETWEEN %s AND %s", $lunes_semana, $domingo_sem);
            break;
        case 'mes':
            $condicion_registros = $wpdb->prepare("AND DATE(entradaR) >= %s", $inicio_mes);
            break;
        case 'todos':
        default:
            $condicion_registros = '';
            break;
    }

    // Cargar registros del usuario
    $registros = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT * FROM $tabla_registro
             WHERE user_id = %d
             $condicion_registros
             ORDER BY entradaR DESC",
            $user_id
        )
    );

    // Agrupar por día
    $dias = [];
    if ($registros) {
        foreach ($registros as $r) {
            $fecha = $r->entradaR ? date('Y-m-d', strtotime($r->entradaR)) : ($r->entrada ? date('Y-m-d', strtotime($r->entrada)) : '');
            if (!$fecha) continue;
            $dias[$fecha][] = $r;
        }
    }

    ob_start();
    echo $mensaje_html;
    ?>

    <form method="get" class="pure-form filtros-container" style="text-align:center; margin-bottom:20px;">
        <?php
        $f_actual = $filtro;
        $boton = function($nombre, $valor, $filtro_actual) {
            $clase = ($filtro_actual === $valor) ? 'pure-button pure-button-primary' : 'pure-button';
            echo "<button type='submit' name='filtro' value='".esc_attr($valor)."' class='$clase'>$nombre</button> ";
        };
        // $boton('Hoy', 'hoy', $f_actual);
        // $boton('Ayer', 'ayer', $f_actual);
        // $boton('Semana Actual', 'semana', $f_actual);
        // $boton('Mes Actual', 'mes', $f_actual);
        // $boton('Todos', 'todos', $f_actual);
        ?>
    </form>

    <h3>Registros (para solicitar cambios)</h3>

    <div style="text-align:right; margin-bottom:10px;">
        <button type="button"
                class="pure-button pure-button-secondary"
                onclick="abrirModalCambioFichaje()">
            ✏️ Solicitar cambio del fichaje seleccionado
        </button>
    </div>
    <div style="overflow-x:auto;">
    <table class="pure-table pure-table-bordered" style="width:100%; text-align:center; table-layout:auto;">
        <thead>
            <tr style="background:#007bff; color:#fff;">
                <th style="width:40px;"></th>
                <th style="min-width:120px;">Día</th>
                <th style="min-width:120px;">Entrada</th>
                <th style="min-width:120px;">Salida</th>
                <th style="min-width:120px;">Tiempo</th>
                <th style="min-width:120px; white-space:nowrap;">Total del día</th>
            </tr>
        </thead>
        <tbody>

        <tbody>
        <?php
        if (!$dias) {
            echo "<tr><td colspan='6'>No hay registros en este periodo.</td></tr>";
        } else {
            $alternar = true;

            foreach ($dias as $fecha => $registros_dia) {
                // Total día
                $total_dia_segundos = 0;
                foreach ($registros_dia as $r) {
                    if ($r->tiempo_totalR) {
                        list($h, $m, $s) = explode(':', $r->tiempo_totalR);
                        $total_dia_segundos += ($h * 3600) + ($m * 60) + $s;
                    }
                }
                $horas_dia    = floor($total_dia_segundos / 3600);
                $minutos_dia  = floor(($total_dia_segundos % 3600) / 60);
                $segundos_dia = $total_dia_segundos % 60;
                $total_dia    = sprintf('%02d:%02d:%02d', $horas_dia, $minutos_dia, $segundos_dia);

                $estilo_total_dia = '';
                if ($total_dia_segundos >= 43200) { // 12h
                    $estilo_total_dia = "style='background-color: #f8d7da; color: red; font-weight: bold;'";
                } elseif ($total_dia_segundos >= 28800) { // 8h
                    $estilo_total_dia = "style='color: red;'";
                }

                $estilo_fila = $alternar ? 'background-color: #f5f5f5;' : 'background-color: #e0e0e0;';
                $alternar = !$alternar;

                $rowspan          = count($registros_dia);
                $primero          = true;
                $fecha_formateada = date('d/m/Y', strtotime($fecha));

                foreach ($registros_dia as $r) {
                    $id_registro     = (int) $r->id;
                    $tiene_pendiente = in_array($id_registro, $registros_pendientes, true);

                    $entrada_dt = $r->entradaR ?: $r->entrada;
                    $salida_dt  = $r->salidaR  ?: $r->salida;

                    $hora_entrada = $entrada_dt ? date('H:i:s', strtotime($entrada_dt)) : '---';
                    $hora_salida  = $salida_dt  ? date('H:i:s', strtotime($salida_dt))  : '---';

                    $tiempo = $r->tiempo_totalR ?: '---';

                    if ($tiene_pendiente) {
                        $celda_sel = "<span style='font-size:11px;color:#b76b00;'>Pendiente</span>";
                    } else {
                        // Guardamos el datetime original en data-* para precargar el modal
                        $celda_sel = "<input type='radio'
                                             name='registro_id_sel'
                                             value='{$id_registro}'
                                             data-entrada='" . esc_attr($entrada_dt) . "'
                                             data-salida='" . esc_attr($salida_dt) . "'>";
                    }

                    echo "<tr style='$estilo_fila'>";

                    // Columna selección
                    echo "<td style='text-align:center; vertical-align:middle;'>{$celda_sel}</td>";

                    if ($primero) {
                        echo "<td rowspan='$rowspan'>$fecha_formateada</td>";
                    }

                    echo "<td>{$hora_entrada}</td>
                          <td>{$hora_salida}</td>
                          <td>{$tiempo}</td>";

                    if ($primero) {
                        echo "<td $estilo_total_dia rowspan='$rowspan'>$total_dia</td>";
                        $primero = false;
                    }

                    echo "</tr>";
                }
            }
        }
        ?>
        </tbody>
    </table>

    <!-- Modal para solicitar cambio de fichaje -->
    <div id="modal-cambio-fichaje"
         style="display:none; position:fixed; inset:0; background:rgba(0,0,0,0.45); z-index:9999; align-items:center; justify-content:center;">
        <div style="background:#fff; padding:20px; border-radius:8px; max-width:400px; width:90%; box-shadow:0 10px 30px rgba(0,0,0,0.2);">
            <h3 style="margin-top:0; margin-bottom:10px;">Solicitar cambio de fichaje</h3>
            <form method="post">
                <?php wp_nonce_field('cambio_fichaje_nonce_action', 'cambio_fichaje_nonce'); ?>
                <input type="hidden" name="accion" value="solicitar_cambio_fichaje">
                <input type="hidden" name="registro_id" id="cambio_registro_id">

                <div style="margin-bottom:10px;">
                    <label style="font-size:13px; color:#555;">Entrada nueva:</label>
                    <input type="datetime-local" name="entrada_nueva" id="cambio_entrada_nueva" required style="width:100%;">
                </div>

                <div style="margin-bottom:10px;">
                    <label style="font-size:13px; color:#555;">Salida nueva (opcional):</label>
                    <input type="datetime-local" name="salida_nueva" id="cambio_salida_nueva" style="width:100%;">
                </div>

                <div style="margin-bottom:10px;">
                    <label style="font-size:13px; color:#555;">Motivo del cambio:</label>
                    <textarea name="motivo" id="cambio_motivo" rows="3" style="width:100%;" required></textarea>
                </div>

                <div style="display:flex; justify-content:flex-end; gap:8px;">
                    <button type="button"
                            class="pure-button"
                            onclick="cerrarModalCambioFichaje()">
                        Cancelar
                    </button>
                    <button type="submit"
                            class="pure-button pure-button-primary">
                        Enviar solicitud
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
    function abrirModalCambioFichaje() {
        const seleccionado = document.querySelector('input[name="registro_id_sel"]:checked');
        if (!seleccionado) {
            alert("Selecciona primero un fichaje de la lista.");
            return;
        }

        const registroId = seleccionado.value;
        const entradaRaw = seleccionado.dataset.entrada || '';
        const salidaRaw  = seleccionado.dataset.salida || '';

        const toInputValue = (str) => {
            if (!str) return '';
            // "YYYY-MM-DD HH:MM:SS" -> "YYYY-MM-DDTHH:MM"
            return str.replace(' ', 'T').slice(0, 16);
        };

        document.getElementById('cambio_registro_id').value   = registroId;
        document.getElementById('cambio_entrada_nueva').value = toInputValue(entradaRaw);
        document.getElementById('cambio_salida_nueva').value  = toInputValue(salidaRaw);
        document.getElementById('cambio_motivo').value        = '';

        const modal = document.getElementById('modal-cambio-fichaje');
        if (modal) {
            modal.style.display = 'flex';
        }
    }

    function cerrarModalCambioFichaje() {
        const modal = document.getElementById('modal-cambio-fichaje');
        if (modal) {
            modal.style.display = 'none';
        }
    }
    </script>

    <?php
    return ob_get_clean();
}
add_shortcode('registro_cambio_horario', 'rh_shortcode_registro_cambio_horario');
