<?php
if (!defined('ABSPATH')) exit;

global $wpdb;
$tabla_empresas = $wpdb->prefix . 'empresas';
$tabla_lugares  = $wpdb->prefix . 'lugares';

$current_page = isset($_GET['page']) ? sanitize_key($_GET['page']) : 'control-horario';
$current_tab  = isset($_GET['tab'])  ? sanitize_key($_GET['tab'])  : 'lugares';

// URL base de la pestaña actual (control-horario & tab=lugares)
$base_url = add_query_arg(
    ['page' => $current_page, 'tab' => $current_tab],
    admin_url('admin.php')
);

if (!current_user_can('manage_options')) {
    wp_die('No tienes permisos para acceder a esta sección.');
}

/* =========================
   Guardar edición
========================= */
if (isset($_POST['guardar_edicion_lugar'])) {
    check_admin_referer('editar_lugar', 'editar_lugar_nonce');

    $id         = intval($_POST['editar_lugar_id']);
    $empresa_id = intval($_POST['empresa_id']);
    $nombre     = sanitize_text_field($_POST['nombre_lugar']);
    $direccion  = sanitize_textarea_field($_POST['direccion']);
    // FIX checkbox (si no está marcado no llega por POST)
    $estado     = (!empty($_POST['estado']) && $_POST['estado'] === 'activado') ? 'activado' : 'desactivado';

    $ok = $wpdb->update(
        $tabla_lugares,
        [
            'empresa_id'   => $empresa_id,
            'nombre_lugar' => $nombre,
            'direccion'    => $direccion,
            'estado'       => $estado,
            'updated_at'   => current_time('mysql'),
        ],
        ['id' => $id]
    );

    if ($ok !== false) {
        // Redirección recomendada para evitar reenvío y mantener pestaña
        wp_safe_redirect($base_url);
        exit;
    } else {
        echo '<div class="error"><p>❌ Error al actualizar: ' . esc_html($wpdb->last_error) . '</p></div>';
    }
}

/* =========================
   Insertar nuevo lugar
========================= */
if (isset($_POST['nuevo_lugar'])) {
    check_admin_referer('nuevo_lugar', 'nuevo_lugar_nonce');

    $empresa_id = intval($_POST['empresa_id']);
    $nombre     = sanitize_text_field($_POST['nombre_lugar']);
    $direccion  = sanitize_textarea_field($_POST['direccion']);

    $ok = $wpdb->insert(
        $tabla_lugares,
        [
            'empresa_id'   => $empresa_id,
            'nombre_lugar' => $nombre,
            'direccion'    => $direccion,
            'estado'       => 'activado',
            'created_at'   => current_time('mysql'),
            'updated_at'   => current_time('mysql'),
        ]
    );

    if ($ok) {
        wp_safe_redirect($base_url);
        exit;
    } else {
        echo '<div class="error"><p>❌ Error al insertar: ' . esc_html($wpdb->last_error) . '</p></div>';
    }
}

/* =========================
   Filtros (GET)
========================= */
$filtro_empresa = isset($_GET['filtro_empresa']) ? intval($_GET['filtro_empresa']) : 0;
$filtro_estado  = isset($_GET['filtro_estado'])  ? sanitize_text_field($_GET['filtro_estado']) : '';

$where = "1=1";
if ($filtro_empresa > 0) {
    $where .= $wpdb->prepare(" AND empresa_id = %d", $filtro_empresa);
}
if (in_array($filtro_estado, ['activado','desactivado'], true)) {
    $where .= $wpdb->prepare(" AND estado = %s", $filtro_estado);
}

/* =========================
   Datos
========================= */
$empresas = $wpdb->get_results("SELECT id, razon_social FROM $tabla_empresas ORDER BY razon_social ASC");
$lugares  = $wpdb->get_results("SELECT * FROM $tabla_lugares WHERE $where ORDER BY created_at DESC");

$lugar_edit = null;
if (isset($_GET['editar']) && is_numeric($_GET['editar'])) {
    $lugar_edit = $wpdb->get_row($wpdb->prepare("SELECT * FROM $tabla_lugares WHERE id = %d", intval($_GET['editar'])));
}
?>

<div class="wrap">
    <h1 class="wp-heading-inline">📍 Gestión de Lugares</h1>

    <?php if ($lugar_edit): ?>
        <hr>
        <h2>Editar lugar</h2>
        <form method="post" action="<?php echo esc_url($base_url . '&editar=' . (int)$lugar_edit->id); ?>">
            <?php wp_nonce_field('editar_lugar', 'editar_lugar_nonce'); ?>
            <input type="hidden" name="editar_lugar_id" value="<?php echo esc_attr($lugar_edit->id); ?>">

            <table class="form-table">
                <tr>
                    <th>Empresa</th>
                    <td>
                        <select name="empresa_id" required>
                            <?php foreach ($empresas as $empresa): ?>
                                <option value="<?php echo esc_attr($empresa->id); ?>" <?php selected($empresa->id, $lugar_edit->empresa_id); ?>>
                                    <?php echo esc_html($empresa->razon_social); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th>Nombre del lugar</th>
                    <td><input type="text" name="nombre_lugar" value="<?php echo esc_attr($lugar_edit->nombre_lugar); ?>" required class="regular-text"></td>
                </tr>
                <tr>
                    <th>Dirección</th>
                    <td><textarea name="direccion" class="large-text" rows="3"><?php echo esc_textarea($lugar_edit->direccion); ?></textarea></td>
                </tr>
                <tr>
                    <th>Activar</th>
                    <td>
                        <input type="checkbox"
                            name="estado"
                            value="activado"
                            <?php checked($lugar_edit->estado, 'activado'); ?> />
                    </td>
                </tr>

            </table>

            <p><input type="submit" name="guardar_edicion_lugar" class="button-primary" value="Guardar cambios"></p>
        </form>
    <?php endif; ?>

    <hr>

    <h2>Añadir nuevo lugar</h2>
    <form method="post" action="<?php echo esc_url($base_url); ?>">
        <?php wp_nonce_field('nuevo_lugar', 'nuevo_lugar_nonce'); ?>
        <table class="form-table">
            <tr>
                <th>Empresa</th>
                <td>
                    <select name="empresa_id" required>
                        <option value="">Selecciona una empresa</option>
                        <?php foreach ($empresas as $empresa): ?>
                            <option value="<?php echo esc_attr($empresa->id); ?>"><?php echo esc_html($empresa->razon_social); ?></option>
                        <?php endforeach; ?>
                    </select>
                </td>
            </tr>
            <tr>
                <th>Nombre del lugar</th>
                <td><input type="text" name="nombre_lugar" required class="regular-text"></td>
            </tr>
            <tr>
                <th>Dirección</th>
                <td><textarea name="direccion" class="large-text" rows="3"></textarea></td>
            </tr>
        </table>
        <p><input type="submit" name="nuevo_lugar" class="button button-primary" value="Añadir lugar"></p>
    </form>

    <hr>

    <h2>Listado de lugares</h2>
    <form method="get" style="margin-bottom:10px;">
        <input type="hidden" name="page" value="<?php echo esc_attr($current_page); ?>">
        <input type="hidden" name="tab"  value="<?php echo esc_attr($current_tab); ?>">

        <select name="filtro_empresa">
            <option value="0">Todas las empresas</option>
            <?php foreach ($empresas as $empresa): ?>
                <option value="<?php echo esc_attr($empresa->id); ?>" <?php selected($filtro_empresa, $empresa->id); ?>>
                    <?php echo esc_html($empresa->razon_social); ?>
                </option>
            <?php endforeach; ?>
        </select>

        <select name="filtro_estado">
            <option value="">Todos los estados</option>
            <option value="activado"   <?php selected($filtro_estado, 'activado');   ?>>Activado</option>
            <option value="desactivado"<?php selected($filtro_estado, 'desactivado');?>>Desactivado</option>
        </select>

        <input type="submit" class="button" value="Filtrar">
    </form>

    <table class="widefat fixed striped">
        <thead>
            <tr>
                <th>ID</th>
                <th>Empresa</th>
                <th>Nombre</th>
                <th>Dirección</th>
                <th>Estado</th>
                <th>Acciones</th>
            </tr>
        </thead>
        <tbody>
        <?php if ($lugares): ?>
            <?php foreach ($lugares as $lugar):
                $empresa_nombre = 'Desconocido';
                foreach ($empresas as $e) { if ((int)$e->id === (int)$lugar->empresa_id) { $empresa_nombre = $e->razon_social; break; } }
                $badge_bg = ($lugar->estado === 'activado') ? '#e6ffed' : '#ffe9e9';
                $badge_fg = ($lugar->estado === 'activado') ? '#0a7' : '#c00';
                $edit_url = add_query_arg(['editar' => $lugar->id], $base_url);
            ?>
                <tr>
                    <td><?php echo (int)$lugar->id; ?></td>
                    <td><?php echo esc_html($empresa_nombre); ?></td>
                    <td><?php echo esc_html($lugar->nombre_lugar); ?></td>
                    <td><?php echo esc_html($lugar->direccion); ?></td>
                    <td>
                        <span style="padding:2px 6px;border-radius:4px;background:<?php echo $badge_bg;?>;color:<?php echo $badge_fg;?>;">
                            <?php echo esc_html($lugar->estado); ?>
                        </span>
                    </td>
                    <td>
                        <a href="<?php echo esc_url($edit_url); ?>" class="button">Editar</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        <?php else: ?>
            <tr><td colspan="6">No hay lugares para los filtros seleccionados.</td></tr>
        <?php endif; ?>
        </tbody>
    </table>
</div>
